# Automatic Image Tagging Setup

## Overview
Every uploaded listing image is automatically analyzed and tagged with detected objects/content using Google Cloud Vision API.

## Features
- **AI-powered tagging**: Detects objects, scenes, activities in photos
- **Confidence scores**: Each tag includes a confidence level (0-1)
- **Fallback analysis**: Basic image properties if API unavailable
- **Automatic**: Tags generated during upload, no manual work needed

## Setup Instructions

### 1. Get Google Cloud Vision API Key

1. Go to [Google Cloud Console](https://console.cloud.google.com/)
2. Create a new project or select existing one
3. Enable **Cloud Vision API**:
   - Navigate to "APIs & Services" > "Library"
   - Search for "Cloud Vision API"
   - Click "Enable"
4. Create API Key:
   - Go to "APIs & Services" > "Credentials"
   - Click "Create Credentials" > "API Key"
   - Copy the generated API key

### 2. Configure Laravel

Add to your `.env` file:
```env
GOOGLE_VISION_API_KEY=your_api_key_here
```

### 3. Run Migration

```bash
php artisan migrate
```

This adds the `tags` JSON column to `listing_media` table.

## How It Works

### Upload Flow
1. User uploads image
2. Image is compressed and watermarked
3. **Google Vision API analyzes image** (detects objects, scenes, etc.)
4. Tags with confidence ≥ 0.7 are stored
5. If API fails/unavailable, basic analysis runs (orientation, resolution)

### Example Tags Output
```json
{
  "tags": [
    {"label": "Car", "confidence": 0.95},
    {"label": "Vehicle", "confidence": 0.92},
    {"label": "Sedan", "confidence": 0.87},
    {"label": "Red", "confidence": 0.78}
  ]
}
```

### Fallback Tags (No API)
If Google Vision API is not configured or fails:
```json
{
  "tags": [
    {"label": "landscape", "confidence": 1.0},
    {"label": "high-resolution", "confidence": 1.0}
  ]
}
```

## API Response Format

### Upload Media Response
```json
{
  "success": true,
  "data": {
    "media": [
      {
        "id": 123,
        "type": "image",
        "path": "listing-media/abc123.jpg",
        "watermarked_path": "listing-media/abc123_watermarked.jpg",
        "tags": [
          {"label": "Smartphone", "confidence": 0.96},
          {"label": "Electronics", "confidence": 0.89}
        ],
        "content_hash": "...",
        "mime_type": "image/jpeg",
        "size_bytes": 45678
      }
    ]
  }
}
```

### Listing Resource
```json
{
  "media": [
    {
      "id": 123,
      "type": "image",
      "path": "listing-media/abc123.jpg",
      "watermarked_path": "listing-media/abc123_watermarked.jpg",
      "tags": [
        {"label": "Laptop", "confidence": 0.94},
        {"label": "Computer", "confidence": 0.91}
      ],
      "is_primary": true
    }
  ]
}
```

## Use Cases

### 1. Search Enhancement
Use tags to improve search results:
```php
// Search listings by image content
$listings = Listing::whereHas('media', function($query) {
    $query->whereJsonContains('tags', [['label' => 'Car']]);
})->get();
```

### 2. Auto-categorization
Suggest categories based on detected objects:
```php
$tags = $media->tags;
$labels = collect($tags)->pluck('label')->toArray();

if (in_array('Car', $labels) || in_array('Vehicle', $labels)) {
    // Suggest "Vehicles" category
}
```

### 3. Content Moderation
Flag inappropriate content:
```php
$inappropriateLabels = ['Weapon', 'Violence', 'Adult content'];
$detectedLabels = collect($media->tags)->pluck('label')->toArray();

if (array_intersect($inappropriateLabels, $detectedLabels)) {
    // Flag for review
}
```

### 4. Display Tags to Users
Show what's in the photo:
```javascript
// Frontend
media.tags.forEach(tag => {
  if (tag.confidence > 0.8) {
    console.log(`Detected: ${tag.label} (${Math.round(tag.confidence * 100)}% confident)`);
  }
});
```

## Cost Considerations

### Google Vision API Pricing (as of 2024)
- First 1,000 images/month: **FREE**
- 1,001 - 5,000,000: $1.50 per 1,000 images
- 5,000,001+: $0.60 per 1,000 images

### Example Monthly Costs
- 500 uploads/month: **FREE**
- 5,000 uploads/month: ~$6
- 50,000 uploads/month: ~$75

## Disabling Auto-Tagging

To disable (use fallback only):
1. Remove `GOOGLE_VISION_API_KEY` from `.env`
2. Images will still get basic tags (orientation, resolution)

## Troubleshooting

### Tags are empty
- Check API key is correct in `.env`
- Verify Cloud Vision API is enabled in Google Cloud Console
- Check Laravel logs: `storage/logs/laravel.log`

### API errors
```bash
# Check logs
tail -f storage/logs/laravel.log | grep "Google Vision"
```

### Testing
```bash
# Upload test image via API
curl -X POST http://your-domain/api/v1/businesses/1/listings/media/upload \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -F "media[]=@test-image.jpg"
```

## Files Modified
- `app/Services/ImageRecognitionService.php` - AI tagging logic
- `app/Services/ImageProcessingService.php` - Integrated tagging
- `app/Models/ListingMedia.php` - Tags field
- `database/migrations/*_add_tags_to_listing_media.php` - Schema
- `config/services.php` - API configuration

## Support
For issues with Google Vision API, see: https://cloud.google.com/vision/docs
