# Shared Hosting Deployment Guide

## Prerequisites
- PHP 8.2+ on your shared hosting
- MySQL/MariaDB database
- File manager or FTP access (FileZilla, cPanel File Manager, etc.)

---

## Step 1: Prepare Files Locally

### Files to Upload:
✅ **Upload these:**
- `app/` (entire directory)
- `bootstrap/` (entire directory)
- `config/` (entire directory)
- `database/` (entire directory)
- `public/` (entire directory)
- `resources/` (entire directory)
- `routes/` (entire directory)
- `storage/` (entire directory)
- `vendor/` (entire directory) ⚠️ **IMPORTANT: Upload this now**
- `composer.json`
- `composer.lock` ⚠️ **IMPORTANT: Upload this now**
- `artisan`
- `.htaccess` (if exists in root)

❌ **DO NOT upload:**
- `.env` (create separately on server)
- `node_modules/`
- `.git/`
- `tests/`
- `.env.example` (optional)

---

## Step 2: Upload to Server

### Option A: Using cPanel File Manager
1. Login to cPanel
2. Go to **File Manager**
3. Navigate to your domain's root (usually `public_html/`)
4. Create a folder for your Laravel app (e.g., `laravel_app/`)
5. Upload all files from Step 1 to this folder
6. Extract any ZIP files if you compressed before uploading

### Option B: Using FTP (FileZilla)
1. Connect to your server via FTP
2. Navigate to `public_html/` or your domain root
3. Create folder `laravel_app/`
4. Upload all files from Step 1

---

## Step 3: Configure Public Directory

Laravel's entry point is `public/index.php`, not the root.

### Method 1: Subdomain (Recommended)
1. In cPanel, go to **Subdomains**
2. Create subdomain: `api.yourdomain.com`
3. Set document root to: `/home/username/public_html/laravel_app/public`

### Method 2: Main Domain
1. Move contents of `public/` to `public_html/`
2. Move Laravel app files to `public_html/laravel_app/`
3. Edit `public_html/index.php`:
   ```php
   require __DIR__.'/../laravel_app/vendor/autoload.php';
   $app = require_once __DIR__.'/../laravel_app/bootstrap/app.php';
   ```

### Method 3: .htaccess Redirect
Create `.htaccess` in `public_html/`:
```apache
<IfModule mod_rewrite.c>
    RewriteEngine On
    RewriteRule ^(.*)$ public/$1 [L]
</IfModule>
```

---

## Step 4: Set Permissions

Using File Manager, set these permissions:

```
storage/ → 755 (recursive)
storage/framework/ → 755 (recursive)
storage/logs/ → 755 (recursive)
bootstrap/cache/ → 755 (recursive)
```

**Right-click folder → Change Permissions → Check: Owner (Read, Write, Execute)**

---

## Step 5: Create .env File

1. In File Manager, navigate to your Laravel root
2. Create new file: `.env`
3. Copy contents from `.env.example`
4. Edit these values:

```env
APP_NAME=Yilick
APP_ENV=production
APP_KEY=
APP_DEBUG=false
APP_URL=https://yourdomain.com

LOG_CHANNEL=stack
LOG_DEPRECATIONS_CHANNEL=null
LOG_LEVEL=error

DB_CONNECTION=mysql
DB_HOST=localhost
DB_PORT=3306
DB_DATABASE=your_database_name
DB_USERNAME=your_database_user
DB_PASSWORD=your_database_password

BROADCAST_DRIVER=log
CACHE_DRIVER=file
FILESYSTEM_DISK=public
QUEUE_CONNECTION=sync
SESSION_DRIVER=file
SESSION_LIFETIME=120

GOOGLE_VISION_API_KEY=your_google_vision_key
OPENAI_API_KEY=your_openai_key
OPENAI_MODEL=gpt-4o-mini

JWT_SECRET=
JWT_ALGO=HS256
JWT_TTL=60
JWT_REFRESH_TTL=20160
```

---

## Step 6: Generate Application Key

### Using PHP CLI (if available):
```bash
cd /home/username/public_html/laravel_app
php artisan key:generate
```

### Without CLI Access:
1. Visit: https://generate-random.org/laravel-key-generator
2. Copy the generated key (e.g., `base64:xxxxx...`)
3. Edit `.env` file
4. Set: `APP_KEY=base64:xxxxx...`

---

## Step 7: Generate JWT Secret

### Using PHP CLI (if available):
```bash
php artisan jwt:secret
```

### Without CLI Access:
1. Generate random 32-character string at: https://www.random.org/strings/
2. Edit `.env` file
3. Set: `JWT_SECRET=your_32_character_random_string`

---

## Step 8: Create Database

1. In cPanel, go to **MySQL Databases**
2. Create new database (e.g., `username_yilick`)
3. Create new user with strong password
4. Add user to database with **ALL PRIVILEGES**
5. Update `.env` with these credentials

---

## Step 9: Import Database

### Option A: Using phpMyAdmin
1. Open phpMyAdmin from cPanel
2. Select your database
3. Click **Import** tab
4. Upload your SQL file
5. Click **Go**

### Option B: Manual Migration (if you have PHP CLI)
```bash
cd /home/username/public_html/laravel_app
php artisan migrate --force
```

---

## Step 10: Create Storage Symlink

### With CLI Access:
```bash
php artisan storage:link
```

### Without CLI Access:
1. In File Manager, go to `public/`
2. Create folder named `storage`
3. This is a workaround - you'll need to manually copy uploaded files

**OR** create a PHP script `create_symlink.php` in `public/`:
```php
<?php
symlink('../storage/app/public', 'storage');
echo 'Symlink created!';
```
Visit: `https://yourdomain.com/create_symlink.php`
Then delete the file.

---

## Step 11: Clear Caches

### With CLI:
```bash
php artisan config:cache
php artisan route:cache
php artisan view:cache
```

### Without CLI:
Create file `clear_cache.php` in `public/`:
```php
<?php
require __DIR__.'/../vendor/autoload.php';
$app = require_once __DIR__.'/../bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);

$kernel->call('config:cache');
$kernel->call('route:cache');
$kernel->call('view:cache');

echo 'Caches cleared and rebuilt!';
```
Visit: `https://yourdomain.com/clear_cache.php`
Then delete the file.

---

## Step 12: Test Your API

Visit: `https://yourdomain.com/api/v1/health`

Expected response:
```json
{
    "status": "healthy",
    "timestamp": "2025-12-21T08:48:00.000000Z"
}
```

---

## Troubleshooting

### Error: "500 Internal Server Error"
- Check `.env` file exists and is configured
- Check `storage/` and `bootstrap/cache/` permissions (755)
- Check error logs in cPanel or `storage/logs/laravel.log`

### Error: "No application encryption key"
- Run Step 6 to generate `APP_KEY`

### Error: "SQLSTATE[HY000] [1045] Access denied"
- Check database credentials in `.env`
- Verify user has privileges on the database

### Error: "Class not found"
- Ensure `vendor/` directory was uploaded
- Ensure `composer.lock` was uploaded

### Images not showing
- Run Step 10 to create storage symlink
- Check `storage/app/public/` permissions

---

## Security Checklist

✅ Set `APP_DEBUG=false` in production
✅ Set `APP_ENV=production`
✅ Use strong database password
✅ Keep `.env` file secure (never commit to git)
✅ Set proper file permissions (755 for directories, 644 for files)
✅ Enable HTTPS/SSL certificate
✅ Disable directory listing in `.htaccess`

---

## Performance Tips

1. **Enable OPcache** (ask hosting provider)
2. **Use route caching**: `php artisan route:cache`
3. **Use config caching**: `php artisan config:cache`
4. **Use view caching**: `php artisan view:cache`
5. **Set `SESSION_DRIVER=database`** for better performance

---

## Need Help?

- Check `storage/logs/laravel.log` for errors
- Contact your hosting provider for PHP CLI access
- Verify PHP version: Create `info.php` with `<?php phpinfo();`
