<?php

namespace App\Console\Commands;

use App\Models\Listing;
use App\Models\UserSubscription;
use Carbon\Carbon;
use Illuminate\Console\Command;

class AutoBumpListings extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'listings:auto-bump';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Automatically bump listings for users with active subscriptions that have auto-bump enabled';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting auto-bump process...');

        $activeSubscriptions = UserSubscription::where('status', 'active')
            ->where('expires_at', '>', now())
            ->with(['package', 'user'])
            ->whereHas('package', function ($query) {
                $query->where('auto_bump_enabled', true);
            })
            ->get();

        $totalBumped = 0;

        foreach ($activeSubscriptions as $subscription) {
            $package = $subscription->package;
            $intervalHours = $package->auto_bump_interval_hours;

            $listings = Listing::where('subscription_id', $subscription->id)
                ->where('status', 'published')
                ->where(function ($query) use ($intervalHours) {
                    $query->whereNull('bumped_at')
                        ->orWhere('bumped_at', '<=', now()->subHours($intervalHours));
                })
                ->get();

            foreach ($listings as $listing) {
                $listing->update([
                    'bumped_at' => now(),
                ]);
                $totalBumped++;
            }

            if ($listings->count() > 0) {
                $this->info("Bumped {$listings->count()} listings for business {$subscription->business_id} (User: {$subscription->user_id}, Package: {$package->name})");
            }
        }

        $this->info("Auto-bump completed. Total listings bumped: {$totalBumped}");

        return Command::SUCCESS;
    }
}
