<?php

namespace App\Console\Commands;

use App\Models\ListingMedia;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;

class CleanupOrphanedListingMedia extends Command
{
    protected $signature = 'listings:cleanup-orphaned-media {--hours=24 : Hours after which orphaned media is deleted}';

    protected $description = 'Delete orphaned listing media (uploaded but never attached to a listing) older than specified hours';

    public function handle(): int
    {
        $hours = (int) $this->option('hours');
        $cutoff = now()->subHours($hours);

        $orphanedMedia = ListingMedia::query()
            ->whereNull('listing_id')
            ->where('created_at', '<', $cutoff)
            ->get();

        if ($orphanedMedia->isEmpty()) {
            $this->info('No orphaned media found older than ' . $hours . ' hours.');
            return self::SUCCESS;
        }

        $count = 0;
        foreach ($orphanedMedia as $media) {
            // Delete file from storage
            if ($media->path && !filter_var($media->path, FILTER_VALIDATE_URL)) {
                Storage::disk('public')->delete($media->path);
            }

            // Delete database record
            $media->delete();
            $count++;
        }

        $this->info("Cleaned up {$count} orphaned media file(s) older than {$hours} hours.");

        return self::SUCCESS;
    }
}
