<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Business;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class BusinessController extends Controller
{
    /**
     * Display a listing of the businesses.
     */
    public function index(Request $request): View
    {
        $query = Business::with(['owner']);

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('primary_phone', 'like', "%{$search}%");
            });
        }

        // Filter by verification status
        if ($request->filled('status')) {
            $query->where('verification_status', $request->status);
        }

        $businesses = $query->latest()->paginate(20)->withQueryString();

        return view('admin.businesses.index', compact('businesses'));
    }

    /**
     * Display the verification queue.
     */
    public function verifications(): View
    {
        $businesses = Business::with(['owner'])
            ->where('verification_status', Business::VERIFICATION_PENDING)
            ->latest()
            ->paginate(20);

        return view('admin.businesses.verifications', compact('businesses'));
    }

    /**
     * Display the specified business.
     */
    public function show(Business $business): View
    {
        $business->load(['owner', 'listings', 'feedbacks']);
        return view('admin.businesses.show', compact('business'));
    }

    /**
     * Approve business verification.
     */
    public function verify(Request $request, Business $business): RedirectResponse
    {
        $business->update([
            'verification_status' => Business::VERIFICATION_VERIFIED,
            'verified_at' => now(),
        ]);

        return back()->with('success', "Business '{$business->name}' has been verified.");
    }

    /**
     * Reject business verification.
     */
    public function reject(Request $request, Business $business): RedirectResponse
    {
        $request->validate([
            'reason' => 'required|string|max:1000',
        ]);

        $business->update([
            'verification_status' => Business::VERIFICATION_REJECTED,
            'verification_rejection_reason' => $request->reason,
            'verified_at' => null,
        ]);

        return back()->with('success', "Verification for '{$business->name}' has been rejected.");
    }
}
