<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;
use Illuminate\Support\Facades\File;

class CategoryController extends Controller
{
    protected string $filePath;

    public function __construct()
    {
        $this->filePath = public_path('assets/categories.json');
    }

    /**
     * Display a listing of categories.
     */
    public function index(): View
    {
        $categories = $this->getCategories();
        return view('admin.categories.index', compact('categories'));
    }

    /**
     * Store a new category.
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'category' => 'required|string|max:255',
            'icon' => 'required|string|max:255',
            'parent' => 'required|string',
        ]);

        $categories = $this->getCategories();
        
        // Generate new ID
        $maxId = collect($categories)->max(fn($c) => (int)$c['id']) ?? 0;
        $newId = (string)($maxId + 1);

        $newCategory = [
            'id' => $newId,
            'Category' => $request->category,
            'icon' => $request->icon,
            'parent' => $request->parent,
            'reg_date' => now()->format('Y-m-d H:i:s'),
        ];

        $categories[] = $newCategory;
        $this->saveCategories($categories);

        return redirect()->route('admin.categories.index')->with('success', 'Category created successfully.');
    }

    /**
     * Update the specified category.
     */
    public function update(Request $request, string $id): RedirectResponse
    {
        $request->validate([
            'category' => 'required|string|max:255',
            'icon' => 'required|string|max:255',
            'parent' => 'required|string',
        ]);

        $categories = $this->getCategories();
        $found = false;

        foreach ($categories as &$category) {
            if ($category['id'] === $id) {
                $category['Category'] = $request->category;
                $category['icon'] = $request->icon;
                $category['parent'] = $request->parent;
                $found = true;
                break;
            }
        }

        if (!$found) {
            return back()->with('error', 'Category not found.');
        }

        $this->saveCategories($categories);

        return redirect()->route('admin.categories.index')->with('success', 'Category updated successfully.');
    }

    /**
     * Remove the specified category.
     */
    public function destroy(string $id): RedirectResponse
    {
        $categories = $this->getCategories();
        
        $filtered = array_filter($categories, fn($c) => $c['id'] !== $id);

        if (count($filtered) === count($categories)) {
            return back()->with('error', 'Category not found.');
        }

        $this->saveCategories(array_values($filtered));

        return redirect()->route('admin.categories.index')->with('success', 'Category deleted successfully.');
    }

    /**
     * Helper to read categories from JSON.
     */
    protected function getCategories(): array
    {
        if (!File::exists($this->filePath)) {
            return [];
        }

        return json_decode(File::get($this->filePath), true) ?? [];
    }

    /**
     * Helper to save categories to JSON.
     */
    protected function saveCategories(array $categories): void
    {
        File::put($this->filePath, json_encode($categories, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
    }
}
