<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Services\FcmService;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class NotificationController extends Controller
{
    protected FcmService $fcmService;

    public function __construct(FcmService $fcmService)
    {
        $this->fcmService = $fcmService;
    }


    /**
     * Display the notification center.
     */
    public function index(): View
    {
        $users = User::has('fcmTokens')->get(['id', 'full_name', 'email']);
        return view('admin.notifications.index', compact('users'));
    }

    /**
     * Send an individual notification.
     */
    public function send(Request $request): RedirectResponse
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'title' => 'required|string|max:255',
            'body' => 'required|string|max:1000',
        ]);

        $user = User::findOrFail($request->user_id);

        try {
            // This handles both Database and FCM (via GeneralNotification's toArray logic)
            $user->notify(new \App\Notifications\GeneralNotification(
                $request->title,
                $request->body,
                ['type' => 'admin_notice'],
                true // Send FCM
            ));

            return back()->with('success', 'Notification sent successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to send notification: ' . $e->getMessage());
        }
    }

    /**
     * Broadcast notification to all users.
     */
    public function broadcast(Request $request): RedirectResponse
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'body' => 'required|string|max:1000',
        ]);

        // 1. Send efficient bulk FCM
        $tokens = \App\Models\UserFcmToken::pluck('fcm_token')->toArray();
        if (!empty($tokens)) {
            $this->fcmService->sendToMultipleTokens(
                $tokens,
                $request->title,
                $request->body,
                ['type' => 'broadcast']
            );
        }

        // 2. Persist to database as a single GLOBAL notification (notifiable_id = null)
        // We manually insert because Notification::route('database') does not support on-demand notifications.
        \Illuminate\Support\Facades\DB::table('notifications')->insert([
            'id' => \Illuminate\Support\Str::uuid(),
            'type' => \App\Notifications\GeneralNotification::class,
            'notifiable_type' => 'App\Models\User', // Polymorphic type matches, but ID is null
            'notifiable_id' => null,
            'data' => json_encode([
                'title' => $request->title,
                'body' => $request->body,
                'data' => ['type' => 'broadcast']
            ]),
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        return back()->with('success', 'Broadcast initialized for ' . count($tokens) . ' devices.');
    }
}
