<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\Business;
use App\Models\Listing;
use App\Models\PosOrder;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class PosController extends Controller
{
    /**
     * Pull updates (Down-sync).
     * Client sends last_sync_timestamp.
     * Sever returns items updated since then.
     */
    public function pull(Request $request, Business $business)
    {
        // Authorize: User must be member of business
        if (!$business->isMember($request->user())) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $lastSync = $request->input('last_sync_timestamp');
        $date = $lastSync ? Carbon::createFromTimestamp($lastSync) : Carbon::create(2000, 1, 1);

        // Fetch updated listings
        $updatedListings = Listing::where('business_id', $business->id)
            ->where(function($q) use ($date) {
                $q->where('updated_at', '>', $date)
                  ->orWhere('created_at', '>', $date);
            })
            ->withTrashed() // Include deleted items to tell POS to remove them
            // Only fetch fields relevant to POS to save bandwidth
            ->get([
                'id', 'business_id', 'title', 'price_amount', 'quantity', 
                'sku', 'barcode', 'stock_alert_level', 'pos_quick_access', 
                'status', 'deleted_at', 'updated_at' // deleted_at needed for removal
            ]);

        // Transform for POS
        $payload = $updatedListings->map(function ($item) {
            return [
                'id' => $item->id,
                'title' => $item->title,
                'price' => $item->price_amount,
                'stock' => $item->quantity,
                'sku' => $item->sku,
                'barcode' => $item->barcode,
                'is_deleted' => !is_null($item->deleted_at),
                'quick_access' => $item->pos_quick_access,
                'updated_at' => $item->updated_at->timestamp,
            ];
        });

        return response()->json([
            'updates' => $payload,
            'timestamp' => now()->timestamp,
        ]);
    }

    /**
     * Push actions (Up-sync).
     * Client sends offline sales/orders.
     */
    public function push(Request $request, Business $business)
    {
        if (!$business->isMember($request->user())) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $validator = Validator::make($request->all(), [
            'orders' => 'required|array',
            'orders.*.offline_uuid' => 'required|string',
            'orders.*.total_amount' => 'numeric',
            'orders.*.items' => 'required|array',
            'orders.*.timestamp' => 'numeric',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $orders = $request->input('orders');
        $processed = [];
        $errors = [];

        DB::beginTransaction();
        try {
            foreach ($orders as $orderData) {
                // Check duplicate by UUID
                if (PosOrder::where('offline_uuid', $orderData['offline_uuid'])->exists()) {
                    $processed[] = $orderData['offline_uuid'];
                    continue;
                }

                // Create Order
                $posOrder = PosOrder::create([
                    'business_id' => $business->id,
                    'cashier_user_id' => $request->user()->id,
                    'total_amount' => $orderData['total_amount'],
                    'items' => $orderData['items'],
                    'payment_method' => $orderData['payment_method'] ?? 'cash',
                    'offline_uuid' => $orderData['offline_uuid'],
                    'synced_at' => now(),
                    'created_at' => Carbon::createFromTimestamp($orderData['timestamp']),
                ]);

                // Update Stock
                foreach ($orderData['items'] as $item) {
                    $listing = Listing::find($item['id']);
                    if ($listing) {
                        // We allow stock to go negative (Physical precedence)
                        // But we log it as "sale_pos"
                        $listing->decrementStock(
                            $item['qty'], 
                            'sale_pos', 
                            'POS-'.$posOrder->id, 
                            $request->user()->id
                        );
                    }
                }
                
                $processed[] = $orderData['offline_uuid'];
            }
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Sync failed', 'error' => $e->getMessage()], 500);
        }

        return response()->json([
            'processed_uuids' => $processed,
            'status' => 'success'
        ]);
    }
}
