<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Resources\ApiResponse;
use App\Models\Ad;
use App\Services\AdTrackerService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class AdController extends Controller
{
    protected AdTrackerService $tracker;

    public function __construct(AdTrackerService $tracker)
    {
        $this->tracker = $tracker;
    }

    /**
     * Get a random ad for a placement
     */
    public function random(Request $request): JsonResponse
    {
        $placement = $request->query('placement');

        $ad = Ad::where('status', 'active')
            ->when($placement, function ($q) use ($placement) {
                $q->where('placement', $placement);
            })
            ->with('sitelinks')
            ->inRandomOrder()
            ->first();

        if (!$ad) {
            return ApiResponse::error('No ads available', 404);
        }

        return ApiResponse::success($ad);
    }

    /**
     * Track an ad event
     */
    public function track(Ad $ad, Request $request): JsonResponse
    {
        $type = $request->input('event');

        if (!in_array($type, ['view', 'click', 'impression'])) {
            return ApiResponse::error('Invalid event type', 422);
        }

        $this->tracker->trackEvent($ad, $type, $request);

        return ApiResponse::success(null, 'Event tracked successfully');
    }
}
