<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ReviewBusinessFeedbackRequest;
use App\Http\Resources\ApiResponse;
use App\Http\Resources\BusinessFeedbackResource;
use App\Models\BusinessFeedback;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class AdminBusinessFeedbackController extends Controller
{
    public function pending(Request $request): JsonResponse
    {
        $perPage = (int) $request->query('per_page', 20);
        $perPage = max(1, min(100, $perPage));

        $query = BusinessFeedback::query()
            ->where('moderation_status', BusinessFeedback::MODERATION_PENDING);

        if ($request->filled('business_id')) {
            $query->where('business_id', (int) $request->query('business_id'));
        }

        if ($request->filled('sentiment')) {
            $query->where('sentiment', $request->query('sentiment'));
        }

        $feedback = $query
            ->with([
                'user',
                'attachments',
                'responses.attachments',
                'responses.respondedBy',
            ])
            ->orderBy('created_at')
            ->paginate($perPage);

        return ApiResponse::success([
            'feedback' => BusinessFeedbackResource::collection($feedback->items()),
            'pagination' => [
                'current_page' => $feedback->currentPage(),
                'per_page' => $feedback->perPage(),
                'total' => $feedback->total(),
                'last_page' => $feedback->lastPage(),
            ],
        ]);
    }

    public function approve(ReviewBusinessFeedbackRequest $request, BusinessFeedback $feedback): JsonResponse
    {
        if ($feedback->moderation_status !== BusinessFeedback::MODERATION_PENDING) {
            return ApiResponse::error('Only pending feedback can be approved.', 422);
        }

        $feedback->update([
            'moderation_status' => BusinessFeedback::MODERATION_APPROVED,
            'reviewed_by_user_id' => auth('api')->id(),
            'reviewed_at' => now(),
            'moderation_notes' => $request->validated()['moderation_notes'] ?? null,
        ]);

        $feedback = BusinessFeedback::query()
            ->whereKey($feedback->getKey())
            ->with([
                'user',
                'attachments',
                'responses.attachments',
                'responses.respondedBy',
            ])
            ->firstOrFail();

        return ApiResponse::success(new BusinessFeedbackResource($feedback), 'Feedback approved successfully.');
    }

    public function reject(ReviewBusinessFeedbackRequest $request, BusinessFeedback $feedback): JsonResponse
    {
        if ($feedback->moderation_status !== BusinessFeedback::MODERATION_PENDING) {
            return ApiResponse::error('Only pending feedback can be rejected.', 422);
        }

        $feedback->update([
            'moderation_status' => BusinessFeedback::MODERATION_REJECTED,
            'reviewed_by_user_id' => auth('api')->id(),
            'reviewed_at' => now(),
            'moderation_notes' => $request->validated()['moderation_notes'] ?? null,
        ]);

        $feedback = BusinessFeedback::query()
            ->whereKey($feedback->getKey())
            ->with([
                'user',
                'attachments',
                'responses.attachments',
                'responses.respondedBy',
            ])
            ->firstOrFail();

        return ApiResponse::success(new BusinessFeedbackResource($feedback), 'Feedback rejected successfully.');
    }
}
