<?php

namespace App\Http\Controllers\Api\V1;

use App\Mail\BusinessInvitationEmail;
use App\Mail\BusinessInvitationStatusEmail;
use App\Mail\BusinessOwnershipTransferredNewOwnerEmail;
use App\Mail\BusinessOwnershipTransferredOldOwnerEmail;
use App\Http\Controllers\Controller;
use App\Http\Requests\Business\CreateBusinessRequest;
use App\Http\Requests\Business\InviteBusinessMemberRequest;
use App\Http\Requests\Business\TransferBusinessOwnershipRequest;
use App\Http\Requests\Business\UpdateBusinessMemberRoleRequest;
use App\Http\Requests\Business\UpdateBusinessRequest;
use App\Http\Resources\ApiResponse;
use App\Http\Resources\BusinessResource;
use App\Http\Resources\UserResource;
use App\Models\Business;
use App\Models\BusinessInvitation;
use App\Models\BusinessMembership;
use App\Models\OtpVerification;
use App\Models\User;
use App\Services\PahappaSmsService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class BusinessController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $user = auth('api')->user();

        $businesses = Business::query()
            ->withCount('followers')
            ->where(function ($q) use ($user) {
                $q->where('owner_user_id', $user->id)
                    ->orWhereHas('memberships', function ($mq) use ($user) {
                        $mq->where('user_id', $user->id);
                    });
            })
            ->orderByDesc('created_at')
            ->get();

        return ApiResponse::success(BusinessResource::collection($businesses));
    }

    public function store(CreateBusinessRequest $request): JsonResponse
    {
        $user = auth('api')->user();

        $max = (int) config('business.max_per_user', 3);
        $ownedCount = Business::where('owner_user_id', $user->id)->count();

        if ($max > 0 && $ownedCount >= $max) {
            return ApiResponse::error('Business creation limit reached.', 403);
        }

        $data = $request->validated();

        if ($request->hasFile('logo')) {
            $data['logo'] = $request->file('logo')->store('business-logos', 'public');
        }

        if ($request->hasFile('cover')) {
            $data['cover'] = $request->file('cover')->store('business-covers', 'public');
        }

        $data['owner_user_id'] = $user->id;

        $business = Business::create($data);

        BusinessMembership::create([
            'business_id' => $business->id,
            'user_id' => $user->id,
            'role' => BusinessMembership::ROLE_OWNER,
            'added_by_user_id' => $user->id,
            'joined_at' => now(),
        ]);

        if ($user->role === User::ROLE_USER) {
            $user->update(['role' => User::ROLE_SELLER]);
        }

        return ApiResponse::created(new BusinessResource($business->fresh()), 'Business created successfully.');
    }

    public function show(Business $business): JsonResponse
    {
        Gate::authorize('view', $business);

        $authUser = auth('api')->user();
        $business = Business::query()
            ->whereKey($business->getKey())
            ->withCount('followers')
            ->withExists(['followers as is_following' => function ($q) use ($authUser) {
                $q->where('users.id', $authUser->id);
            }])
            ->firstOrFail();

        return ApiResponse::success(new BusinessResource($business));
    }

    public function myFollowing(Request $request): JsonResponse
    {
        $user = auth('api')->user();

        $businesses = $user->followedBusinesses()
            ->withCount('followers')
            ->withExists(['followers as is_following' => function ($q) use ($user) {
                $q->where('users.id', $user->id);
            }])
            ->orderByPivot('created_at', 'desc')
            ->get();

        return ApiResponse::success(BusinessResource::collection($businesses));
    }

    public function follow(Business $business): JsonResponse
    {
        $user = auth('api')->user();

        $business->followers()->syncWithoutDetaching([$user->id]);

        $fresh = Business::query()
            ->whereKey($business->getKey())
            ->withCount('followers')
            ->withExists(['followers as is_following' => function ($q) use ($user) {
                $q->where('users.id', $user->id);
            }])
            ->firstOrFail();

        return ApiResponse::success(new BusinessResource($fresh), 'Business followed successfully.');
    }

    public function unfollow(Business $business): JsonResponse
    {
        $user = auth('api')->user();

        $business->followers()->detach($user->id);

        $fresh = Business::query()
            ->whereKey($business->getKey())
            ->withCount('followers')
            ->withExists(['followers as is_following' => function ($q) use ($user) {
                $q->where('users.id', $user->id);
            }])
            ->firstOrFail();

        return ApiResponse::success(new BusinessResource($fresh), 'Business unfollowed successfully.');
    }

    public function update(UpdateBusinessRequest $request, Business $business): JsonResponse
    {
        Gate::authorize('update', $business);

        $data = $request->validated();

        if (isset($data['primary_phone']) && $data['primary_phone'] !== $business->primary_phone) {
            $data['is_primary_phone_verified'] = false;
            $data['primary_phone_verified_at'] = null;
        }

        if (isset($data['secondary_phone']) && $data['secondary_phone'] !== $business->secondary_phone) {
            $data['is_secondary_phone_verified'] = false;
            $data['secondary_phone_verified_at'] = null;
        }

        if (isset($data['primary_whatsapp']) && $data['primary_whatsapp'] !== $business->primary_whatsapp) {
            $data['is_primary_whatsapp_verified'] = false;
            $data['primary_whatsapp_verified_at'] = null;
        }

        if (isset($data['secondary_whatsapp']) && $data['secondary_whatsapp'] !== $business->secondary_whatsapp) {
            $data['is_secondary_whatsapp_verified'] = false;
            $data['secondary_whatsapp_verified_at'] = null;
        }

        if ($request->hasFile('logo')) {
            if ($business->logo && !filter_var($business->logo, FILTER_VALIDATE_URL)) {
                Storage::disk('public')->delete($business->logo);
            }

            $data['logo'] = $request->file('logo')->store('business-logos', 'public');
        }

        if ($request->hasFile('cover')) {
            if ($business->cover && !filter_var($business->cover, FILTER_VALIDATE_URL)) {
                Storage::disk('public')->delete($business->cover);
            }

            $data['cover'] = $request->file('cover')->store('business-covers', 'public');
        }

        $business->update($data);

        return ApiResponse::success(new BusinessResource($business->fresh()), 'Business updated successfully.');
    }

    public function destroy(Business $business): JsonResponse
    {
        Gate::authorize('delete', $business);

        $business->delete();

        return ApiResponse::success(null, 'Business deleted successfully.');
    }

    public function members(Business $business): JsonResponse
    {
        Gate::authorize('view', $business);

        $authUserId = auth('api')->id();
        $isManager = $business->memberships()
            ->where('user_id', $authUserId)
            ->whereIn('role', [
                BusinessMembership::ROLE_OWNER,
                BusinessMembership::ROLE_MANAGER,
                BusinessMembership::ROLE_LISTINGS_MANAGER
            ])
            ->exists();

        $members = $business->members()
            ->select('users.*')
            ->get()
            ->map(function (User $user) use ($business, $isManager) {
                $membership = $business->memberships()->where('user_id', $user->id)->first();
                
                $userData = (new UserResource($user))->resolve();
                
                // If requester is a manager/owner, ensure we show contact info
                if ($isManager) {
                    $userData['email'] = $user->email;
                    $userData['phone'] = $user->phone;
                    $userData['whatsapp_number'] = $user->whatsapp_number;
                }

                return [
                    'user' => $userData,
                    'role' => $membership?->role,
                    'joined_at' => $membership?->joined_at?->toIso8601String(),
                ];
            })
            ->values();

        return ApiResponse::success($members);
    }

    public function invitations(Business $business): JsonResponse
    {
        Gate::authorize('view', $business);

        $invitations = BusinessInvitation::where('business_id', $business->id)
            ->whereNull('accepted_at')
            ->whereNull('revoked_at')
            ->where(function ($q) {
                $q->whereNull('expires_at')->orWhere('expires_at', '>', now());
            })
            ->orderByDesc('created_at')
            ->get();

        return ApiResponse::success($invitations);
    }

    public function followers(Request $request, Business $business): JsonResponse
    {
        Gate::authorize('view', $business);

        $perPage = (int) $request->query('per_page', 20);
        $perPage = max(1, min(100, $perPage));

        $followers = $business->followers()
            ->select('users.*')
            ->orderByDesc('business_follows.created_at')
            ->paginate($perPage);

        return ApiResponse::success([
            'followers' => UserResource::collection($followers->items()),
            'pagination' => [
                'current_page' => $followers->currentPage(),
                'per_page' => $followers->perPage(),
                'total' => $followers->total(),
                'last_page' => $followers->lastPage(),
            ],
        ]);
    }

    public function invite(InviteBusinessMemberRequest $request, Business $business): JsonResponse
    {
        Gate::authorize('invite', $business);

        $email = strtolower(trim($request->email));

        if (strtolower(auth('api')->user()->email) === $email) {
            return ApiResponse::error('You cannot invite yourself.', 400);
        }

        $existingUser = User::where('email', $email)->first();
        if (!$existingUser) {
            return ApiResponse::error('Invited user must already have an account.', 422);
        }

        $alreadyMember = $business->memberships()->where('user_id', $existingUser->id)->exists();
        if ($alreadyMember) {
            return ApiResponse::error('User is already a member of this business.', 400);
        }

        $existingInvite = BusinessInvitation::where('business_id', $business->id)
            ->where('email', $email)
            ->whereNull('accepted_at')
            ->whereNull('revoked_at')
            ->where(function ($q) {
                $q->whereNull('expires_at')->orWhere('expires_at', '>', now());
            })
            ->first();

        if ($existingInvite) {
            return ApiResponse::error('An active invitation already exists for this user.', 400);
        }

        $invite = BusinessInvitation::create([
            'business_id' => $business->id,
            'invited_by_user_id' => auth('api')->id(),
            'email' => $email,
            'role' => $request->role,
            'expires_at' => now()->addDays(7),
        ]);

        try {
            Mail::to($existingUser->email)->send(new BusinessInvitationEmail(
                $business,
                $invite,
                auth('api')->user(),
                $existingUser
            ));
        } catch (\Throwable $e) {
            \Log::warning('Failed to send business invitation email', [
                'business_id' => $business->id,
                'invite_id' => $invite->id,
                'email' => $email,
                'error' => $e->getMessage(),
            ]);
        }

        return ApiResponse::created([
            'id' => $invite->id,
            'email' => $invite->email,
            'role' => $invite->role,
            'token' => $invite->token,
            'expires_at' => $invite->expires_at?->toIso8601String(),
        ], 'Invitation created successfully.');
    }

    public function acceptInvite(Request $request, string $token): JsonResponse
    {
        $user = auth('api')->user();

        $invite = BusinessInvitation::where('token', $token)->first();

        if (!$invite || !$invite->isActive()) {
            return ApiResponse::error('Invalid or expired invitation.', 400);
        }

        if (strtolower($user->email) !== strtolower($invite->email)) {
            return ApiResponse::forbidden('This invitation is not for your account.');
        }

        $business = $invite->business;

        $membership = BusinessMembership::firstOrCreate([
            'business_id' => $business->id,
            'user_id' => $user->id,
        ], [
            'role' => $invite->role,
            'added_by_user_id' => $invite->invited_by_user_id,
            'joined_at' => now(),
        ]);

        if ($invite->accepted_at === null) {
            $invite->update(['accepted_at' => now()]);

            $inviter = $invite->invitedBy;
            if (!$inviter && $invite->invited_by_user_id) {
                $inviter = User::find($invite->invited_by_user_id);
            }

            if ($inviter && $inviter->email) {
                try {
                    Mail::to($inviter->email)->send(new BusinessInvitationStatusEmail(
                        $business,
                        $invite,
                        $inviter,
                        $user,
                        'accepted',
                        'inviter'
                    ));
                } catch (\Throwable $e) {
                    \Log::warning('Failed to send business invite status email (inviter, accepted)', [
                        'business_id' => $business->id,
                        'invite_id' => $invite->id,
                        'inviter_id' => $inviter->id,
                        'invitee_id' => $user->id,
                        'error' => $e->getMessage(),
                    ]);
                }
            }

            if ($user->email) {
                try {
                    Mail::to($user->email)->send(new BusinessInvitationStatusEmail(
                        $business,
                        $invite,
                        $inviter ?? $user,
                        $user,
                        'accepted',
                        'invitee'
                    ));
                } catch (\Throwable $e) {
                    \Log::warning('Failed to send business invite status email (invitee, accepted)', [
                        'business_id' => $business->id,
                        'invite_id' => $invite->id,
                        'inviter_id' => $inviter?->id,
                        'invitee_id' => $user->id,
                        'error' => $e->getMessage(),
                    ]);
                }
            }
        }

        return ApiResponse::success(new BusinessResource($business), 'Invitation accepted successfully.');
    }

    public function declineInvite(Request $request, string $token): JsonResponse
    {
        $user = auth('api')->user();

        $invite = BusinessInvitation::where('token', $token)->first();

        if (!$invite) {
            return ApiResponse::error('Invalid or expired invitation.', 400);
        }

        if (strtolower($user->email) !== strtolower($invite->email)) {
            return ApiResponse::forbidden('This invitation is not for your account.');
        }

        if ($invite->accepted_at !== null) {
            return ApiResponse::error('Invitation has already been accepted.', 400);
        }

        if ($invite->revoked_at !== null) {
            return ApiResponse::success(null, 'Invitation declined successfully.');
        }

        if ($invite->isExpired()) {
            return ApiResponse::error('Invalid or expired invitation.', 400);
        }

        $invite->update(['revoked_at' => now()]);

        $business = $invite->business;
        $inviter = $invite->invitedBy;
        if (!$inviter && $invite->invited_by_user_id) {
            $inviter = User::find($invite->invited_by_user_id);
        }

        if ($inviter && $inviter->email) {
            try {
                Mail::to($inviter->email)->send(new BusinessInvitationStatusEmail(
                    $business,
                    $invite,
                    $inviter,
                    $user,
                    'declined',
                    'inviter'
                ));
            } catch (\Throwable $e) {
                \Log::warning('Failed to send business invite status email (inviter, declined)', [
                    'business_id' => $business?->id,
                    'invite_id' => $invite->id,
                    'inviter_id' => $inviter->id,
                    'invitee_id' => $user->id,
                    'error' => $e->getMessage(),
                ]);
            }
        }

        if ($user->email) {
            try {
                Mail::to($user->email)->send(new BusinessInvitationStatusEmail(
                    $business,
                    $invite,
                    $inviter ?? $user,
                    $user,
                    'declined',
                    'invitee'
                ));
            } catch (\Throwable $e) {
                \Log::warning('Failed to send business invite status email (invitee, declined)', [
                    'business_id' => $business?->id,
                    'invite_id' => $invite->id,
                    'inviter_id' => $inviter?->id,
                    'invitee_id' => $user->id,
                    'error' => $e->getMessage(),
                ]);
            }
        }

        return ApiResponse::success(null, 'Invitation declined successfully.');
    }

    public function updateMemberRole(UpdateBusinessMemberRoleRequest $request, Business $business, int $userId): JsonResponse
    {
        Gate::authorize('manageMembers', $business);

        if ((int) $business->owner_user_id === (int) $userId) {
            return ApiResponse::error('Cannot change the owner role.', 400);
        }

        $membership = BusinessMembership::where('business_id', $business->id)
            ->where('user_id', $userId)
            ->first();

        if (!$membership) {
            return ApiResponse::notFound('Member not found.');
        }

        $membership->update(['role' => $request->role]);

        return ApiResponse::success(null, 'Member role updated successfully.');
    }

    public function removeMember(Business $business, int $userId): JsonResponse
    {
        Gate::authorize('manageMembers', $business);

        if ((int) $business->owner_user_id === (int) $userId) {
            return ApiResponse::error('Cannot remove the owner from the business.', 400);
        }

        $membership = BusinessMembership::where('business_id', $business->id)
            ->where('user_id', $userId)
            ->first();

        if (!$membership) {
            return ApiResponse::notFound('Member not found.');
        }

        $membership->delete();

        return ApiResponse::success(null, 'Member removed successfully.');
    }

    public function transferOwnership(TransferBusinessOwnershipRequest $request, Business $business): JsonResponse
    {
        Gate::authorize('transferOwnership', $business);

        $newOwnerEmail = strtolower(trim($request->new_owner_email));
        $newOwner = User::where('email', $newOwnerEmail)->first();

        if (!$newOwner) {
            return ApiResponse::error('New owner must already have an account.', 422);
        }

        $newOwnerId = (int) $newOwner->id;

        if ($newOwnerId === (int) $business->owner_user_id) {
            return ApiResponse::error('User is already the owner.', 400);
        }

        $previousOwner = $business->owner;
        $oldOwnerId = (int) $business->owner_user_id;

        $newOwnerMembership = BusinessMembership::firstOrCreate([
            'business_id' => $business->id,
            'user_id' => $newOwnerId,
        ], [
            'role' => BusinessMembership::ROLE_OWNER,
            'added_by_user_id' => auth('api')->id(),
            'joined_at' => now(),
        ]);

        $newOwnerMembership->update(['role' => BusinessMembership::ROLE_OWNER]);

        BusinessMembership::where('business_id', $business->id)
            ->where('user_id', $oldOwnerId)
            ->update(['role' => BusinessMembership::ROLE_MANAGER]);

        $business->update(['owner_user_id' => $newOwnerId]);

        $actor = auth('api')->user();
        $freshBusiness = $business->fresh();

        if ($previousOwner && $previousOwner->email) {
            try {
                Mail::to($previousOwner->email)->send(new BusinessOwnershipTransferredOldOwnerEmail(
                    $freshBusiness,
                    $previousOwner,
                    $newOwner,
                    $actor
                ));
            } catch (\Throwable $e) {
                \Log::warning('Failed to send business ownership transfer email (previous owner)', [
                    'business_id' => $freshBusiness->id,
                    'previous_owner_id' => $previousOwner->id,
                    'new_owner_id' => $newOwner->id,
                    'error' => $e->getMessage(),
                ]);
            }
        }

        if ($newOwner->email) {
            try {
                Mail::to($newOwner->email)->send(new BusinessOwnershipTransferredNewOwnerEmail(
                    $freshBusiness,
                    $newOwner,
                    $previousOwner ?? $actor,
                    $actor
                ));
            } catch (\Throwable $e) {
                \Log::warning('Failed to send business ownership transfer email (new owner)', [
                    'business_id' => $freshBusiness->id,
                    'previous_owner_id' => $previousOwner?->id,
                    'new_owner_id' => $newOwner->id,
                    'error' => $e->getMessage(),
                ]);
            }
        }

        return ApiResponse::success(new BusinessResource($freshBusiness), 'Ownership transferred successfully.');
    }

    public function requestVerification(Business $business): JsonResponse
    {
        Gate::authorize('requestVerification', $business);

        if ($business->verification_status === Business::VERIFICATION_VERIFIED) {
            return ApiResponse::error('Business is already verified.', 400);
        }

        $business->requestVerification();

        return ApiResponse::success(new BusinessResource($business->fresh()), 'Verification request submitted.');
    }

    public function sendContactOtp(Request $request, Business $business): JsonResponse
    {
        Gate::authorize('update', $business);

        $validator = Validator::make($request->all(), [
            'type' => ['required', 'string', 'in:' . implode(',', [
                OtpVerification::TYPE_PHONE,
                OtpVerification::TYPE_PHONE_PRIMARY,
                OtpVerification::TYPE_PHONE_SECONDARY,
                OtpVerification::TYPE_WHATSAPP,
                OtpVerification::TYPE_WHATSAPP_PRIMARY,
                OtpVerification::TYPE_WHATSAPP_SECONDARY,
            ])],
        ]);

        if ($validator->fails()) {
            return ApiResponse::validationError($validator->errors()->toArray());
        }

        $type = $validator->validated()['type'];

        $identifier = match ($type) {
            OtpVerification::TYPE_PHONE, OtpVerification::TYPE_PHONE_PRIMARY => $business->primary_phone,
            OtpVerification::TYPE_PHONE_SECONDARY => $business->secondary_phone,
            OtpVerification::TYPE_WHATSAPP, OtpVerification::TYPE_WHATSAPP_PRIMARY => $business->primary_whatsapp,
            OtpVerification::TYPE_WHATSAPP_SECONDARY => $business->secondary_whatsapp,
            default => null,
        };

        if (!$identifier) {
            return ApiResponse::error('No contact number is set for this verification type.', 422);
        }

        $phoneValidator = Validator::make(['identifier' => $identifier], [
            'identifier' => ['required', 'string', 'regex:/^\+?[0-9]{10,15}$/'],
        ]);

        if ($phoneValidator->fails()) {
            return ApiResponse::error('Invalid phone number. Use international format like +2567XXXXXXXX.', 422);
        }

        if (OtpVerification::hasPending($identifier, $type)) {
            return ApiResponse::error('An OTP was recently sent. Please wait before requesting a new one.', 429);
        }

        $otpData = OtpVerification::generateFor($identifier, $type);
        $message = 'Your ' . config('app.name') . ' verification code is ' . $otpData['otp'] . '. It expires in ' . (int) ceil($otpData['expires_in'] / 60) . ' minutes.';

        try {
            $sms = app(PahappaSmsService::class);
            $ok = $sms->send($identifier, $message);

            if (!$ok) {
                throw new \RuntimeException('SMS gateway returned non-ok response.');
            }
        } catch (\Throwable $e) {
            OtpVerification::where('identifier', $identifier)
                ->where('type', $type)
                ->whereNull('verified_at')
                ->delete();

            \Log::warning('Failed to send business contact OTP', [
                'business_id' => $business->id,
                'identifier' => $identifier,
                'type' => $type,
                'error' => $e->getMessage(),
            ]);

            return ApiResponse::error('Unable to send OTP. Please try again later.', 500);
        }

        $responseData = [
            'expires_in' => $otpData['expires_in'],
        ];

        if (config('app.debug')) {
            $responseData['otp'] = $otpData['otp'];
        }

        return ApiResponse::success($responseData, 'OTP sent successfully.');
    }

    public function verifyContactOtp(Request $request, Business $business): JsonResponse
    {
        Gate::authorize('update', $business);

        $validator = Validator::make($request->all(), [
            'type' => ['required', 'string', 'in:' . implode(',', [
                OtpVerification::TYPE_PHONE,
                OtpVerification::TYPE_PHONE_PRIMARY,
                OtpVerification::TYPE_PHONE_SECONDARY,
                OtpVerification::TYPE_WHATSAPP,
                OtpVerification::TYPE_WHATSAPP_PRIMARY,
                OtpVerification::TYPE_WHATSAPP_SECONDARY,
            ])],
            'otp' => ['required', 'digits:6'],
        ]);

        if ($validator->fails()) {
            return ApiResponse::validationError($validator->errors()->toArray());
        }

        $type = $validator->validated()['type'];
        $otp = $validator->validated()['otp'];

        $identifier = match ($type) {
            OtpVerification::TYPE_PHONE, OtpVerification::TYPE_PHONE_PRIMARY => $business->primary_phone,
            OtpVerification::TYPE_PHONE_SECONDARY => $business->secondary_phone,
            OtpVerification::TYPE_WHATSAPP, OtpVerification::TYPE_WHATSAPP_PRIMARY => $business->primary_whatsapp,
            OtpVerification::TYPE_WHATSAPP_SECONDARY => $business->secondary_whatsapp,
            default => null,
        };

        if (!$identifier) {
            return ApiResponse::error('No contact number is set for this verification type.', 422);
        }

        if (!OtpVerification::verify($identifier, $type, $otp)) {
            return ApiResponse::error('Invalid or expired OTP.', 400);
        }

        $now = $business->freshTimestamp();
        $updateData = match ($type) {
            OtpVerification::TYPE_PHONE, OtpVerification::TYPE_PHONE_PRIMARY => [
                'is_primary_phone_verified' => true,
                'primary_phone_verified_at' => $now,
            ],
            OtpVerification::TYPE_PHONE_SECONDARY => [
                'is_secondary_phone_verified' => true,
                'secondary_phone_verified_at' => $now,
            ],
            OtpVerification::TYPE_WHATSAPP, OtpVerification::TYPE_WHATSAPP_PRIMARY => [
                'is_primary_whatsapp_verified' => true,
                'primary_whatsapp_verified_at' => $now,
            ],
            OtpVerification::TYPE_WHATSAPP_SECONDARY => [
                'is_secondary_whatsapp_verified' => true,
                'secondary_whatsapp_verified_at' => $now,
            ],
            default => [],
        };

        $business->forceFill($updateData)->save();

        return ApiResponse::success(new BusinessResource($business->fresh()), 'OTP verified successfully.');
    }
}
