<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Resources\ApiResponse;
use App\Http\Resources\ListingResource;
use App\Models\Listing;
use App\Services\ListingViewTrackerService;
use App\Services\SearchTrackingService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class ListingPublicController extends Controller
{
    public function index(Request $request, SearchTrackingService $searchTracker): JsonResponse
    {
        $perPage = (int) $request->query('per_page', 20);
        $perPage = max(1, min(100, $perPage));

        $query = Listing::query()
            ->publicVisible()
            ->withoutBlockedContent($request->user('api'));
        
        $searchQuery = null;
        $filters = [];

        if ($request->filled('q')) {
            $q = $request->query('q');
            $searchQuery = $q;
            $query->search($q);
        }

        if ($request->filled('category_id')) {
            // Backward compat: ignore category_id (removed). Use category/sub_category/sub_sub_category instead.
        }

        if ($request->filled('category')) {
            $cat = $request->query('category');
            $filters['category'] = $cat;
            $query->where(function ($q) use ($cat) {
                $q->where('category', 'like', $cat)
                    ->orWhere('sub_category', 'like', $cat)
                    ->orWhere('sub_sub_category', 'like', $cat);
            });
        }

        if ($request->filled('sub_category') || $request->filled('subcategory')) {
            $subCat = $request->query('sub_category') ?? $request->query('subcategory');
            $filters['sub_category'] = $subCat;
            $query->where(function ($q) use ($subCat) {
                $q->where('category', 'like', $subCat)
                    ->orWhere('sub_category', 'like', $subCat)
                    ->orWhere('sub_sub_category', 'like', $subCat);
            });
        }

        if ($request->filled('sub_sub_category')) {
            $subSubCat = $request->query('sub_sub_category');
            $filters['sub_sub_category'] = $subSubCat;
            $query->where(function ($q) use ($subSubCat) {
                $q->where('category', 'like', $subSubCat)
                    ->orWhere('sub_category', 'like', $subSubCat)
                    ->orWhere('sub_sub_category', 'like', $subSubCat);
            });
        }

        if ($request->filled('custom')) {
            $custom = $request->query('custom');
            if (is_array($custom)) {
                foreach ($custom as $key => $value) {
                    if (is_string($value) && str_contains($value, ',')) {
                        $value = explode(',', $value);
                    }

                    if (is_array($value)) {
                        $query->where(function ($q) use ($key, $value) {
                            foreach ($value as $v) {
                                $q->orWhereJsonContains('custom_fields->' . $key, $v);
                                // Also handle if custom_fields is a simple string for this key
                                $q->orWhere('custom_fields->' . $key, $v);
                            }
                        });
                    } else {
                        $query->where(function ($q) use ($key, $value) {
                            $q->whereJsonContains('custom_fields->' . $key, $value)
                                ->orWhere('custom_fields->' . $key, $value);
                        });
                    }
                }
            }
        }

        if ($request->filled('tag')) {
            $tag = $request->query('tag');
            $query->whereJsonContains('tags', $tag);
        }

        if ($request->filled('min_price')) {
            $filters['min_price'] = $request->query('min_price');
            $query->where('price_amount', '>=', (float) $request->query('min_price'));
        }

        if ($request->filled('max_price')) {
            $filters['max_price'] = $request->query('max_price');
            $query->where('price_amount', '<=', (float) $request->query('max_price'));
        }

        if ($request->filled('city')) {
            $filters['city'] = $request->query('city');
            $query->where('location_city', $request->query('city'));
        }

        if ($request->filled('district')) {
            $filters['district'] = $request->query('district');
            $query->where('location_district', $request->query('district'));
        }

        if ($request->filled('business_id')) {
            $filters['business_id'] = $request->query('business_id');
            $query->where('business_id', $request->query('business_id'));
        }

        if ($request->filled('country_code')) {
            $filters['country_code'] = $request->query('country_code');
            $query->where('location_country_code', $request->query('country_code'));
        }

        if ($request->boolean('hot_deals')) {
            $query->where('hot_deals', true);
        }

        $listingsQuery = $query
            ->with(['business' => function($q) {
                $q->withCount(['followers', 'listings']);
            }, 'media', 'createdBy']);

        if ($searchQuery) {
            // Prioritize relevance score for searches
            $listings = $listingsQuery
                ->orderByRaw('relevance DESC')
                ->orderByRaw("(promoted_until IS NOT NULL AND promoted_until > NOW()) DESC")
                ->paginate($perPage);
        } else {
            // Default chronological/promoted feed
            $listings = $listingsQuery
                ->orderedForFeed()
                ->paginate($perPage);
        }

        // --- "Did you mean?" Suggestion Logic ---
        $suggestion = null;
        if ($searchQuery && $listings->total() > 0) {
            // If the query was fuzzy (Sequence-LIKE found it), find the most common word in the top results' titles
            // that starts with or is similar to the search query.
            $topListing = $listings->first();
            // If relevance is 0 or very low, it means it was a fuzzy/Soundex match
            if (!$topListing->relevance || $topListing->relevance < 1.0) {
                // Try to extract the best matching word from the title
                $titleWords = explode(' ', strtolower($topListing->title));
                $searchLower = strtolower($searchQuery);
                
                // Simple heuristic: find the word that contains most characters of the search query in order
                foreach ($titleWords as $word) {
                    $word = preg_replace('/[^a-z0-9]/', '', $word);
                    if (strlen($word) < 3) continue;
                    
                    // If word starts with the same char and is reasonably close in length
                    if ($word[0] == $searchLower[0] && abs(strlen($word) - strlen($searchLower)) <= 2) {
                        $suggestion = $word;
                        break;
                    }
                }
            }
        }

        if ($searchQuery || !empty($filters)) {
            $searchTracker->trackSearch(
                $request,
                'text',
                $searchQuery,
                !empty($filters) ? $filters : null,
                $listings->total()
            );
        }

        return ApiResponse::success([
            'listings' => ListingResource::collection($listings->items()),
            'suggestion' => $suggestion,
            'pagination' => [
                'current_page' => $listings->currentPage(),
                'per_page' => $listings->perPage(),
                'total' => $listings->total(),
                'last_page' => $listings->lastPage(),
            ],
        ]);
    }

    public function showBySlug(string $slug, Request $request, ListingViewTrackerService $viewTracker): JsonResponse
    {
        $listing = Listing::query()
            ->publicVisible()
            ->withoutBlockedContent($request->user('api'))
            ->where('slug', $slug)
            ->with(['business' => function($q) {
                $q->withCount(['followers', 'listings']);
            }, 'media', 'createdBy'])
            ->firstOrFail();

        // Track view
        $viewTracker->trackView($listing, $request);

        // Fetch related listings from same sub_category
        $relatedListings = collect();
        if ($listing->sub_category) {
            $relatedListings = Listing::query()
                ->publicVisible()
                ->withoutBlockedContent($request->user('api'))
                ->where('sub_category', $listing->sub_category)
                ->where('id', '!=', $listing->id)
                ->with(['business', 'media', 'createdBy'])
                ->orderedForFeed()
                ->limit(10)
                ->get();
        }

        return ApiResponse::success([
            'listing' => new ListingResource($listing->fresh(['business', 'media', 'createdBy'])),
            'related_listings' => ListingResource::collection($relatedListings),
        ]);
    }
}
