<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\UserFcmToken;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class NotificationController extends Controller
{
    /**
     * Update the user's FCM token.
     */
    public function updateToken(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'fcm_token' => 'required|string',
            'device_id' => 'required|string', // Device identifier from the app
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = $request->user();

        // Update or create the token for this specific device
        UserFcmToken::updateOrCreate(
            [
                'user_id' => $user->id,
                'device_id' => $request->device_id,
            ],
            [
                'fcm_token' => $request->fcm_token,
                'last_used_at' => now(),
            ]
        );

        return response()->json([
            'status' => 'success',
            'message' => 'FCM token registered successfully'
        ]);
    }

    /**
     * Get user notifications.
     */
    public function index(Request $request)
    {
        // Fetch user specific + global notifications
        $notifications = \App\Notifications\GeneralNotification::all() // Placeholder logic, actual query below
            ->where(function($q) use ($request) {
                $q->where('notifiable_id', $request->user()->id)
                  ->where('notifiable_type', get_class($request->user()));
            })
            ->orWhereNull('notifiable_id')
            ->latest()
            ->paginate(20);

        // Since we can't query polymorphic + null efficiently with standard relationship, 
        // we use the DB query builder or a raw query, or direct model usage if available.
        // Assuming 'DatabaseNotification' model is standard:
        $notifications = \Illuminate\Notifications\DatabaseNotification::query()
            ->where(function ($query) use ($request) {
                $query->where('notifiable_id', $request->user()->id)
                      ->where('notifiable_type', get_class($request->user()));
            })
            ->orWhereNull('notifiable_id')
            ->latest()
            ->paginate(20);

        return response()->json($notifications);
    }

    /**
     * Mark notification as read.
     */
    public function markAsRead(Request $request, $id)
    {
        $notification = $request->user()
            ->notifications()
            ->where('id', $id)
            ->first();

        if ($notification) {
            $notification->markAsRead();
        }

        return response()->json(['status' => 'success']);
    }

    /**
     * Mark all notifications as read.
     */
    public function markAllAsRead(Request $request)
    {
        $request->user()->unreadNotifications->markAsRead();
        return response()->json(['status' => 'success']);
    }
}
