<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\Business;
use App\Models\PosOrder;
use App\Models\PosPayment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PosAnalyticsController extends Controller
{
    /**
     * Get sales dashboard summary.
     */
    public function dashboard(Request $request, Business $business)
    {
        if (!$business->isMember($request->user())) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $period = $request->input('period', 'today'); // today, week, month, year

        $query = PosOrder::where('business_id', $business->id);

        if ($period === 'today') {
            $query->whereDate('created_at', today());
        } elseif ($period === 'week') {
            $query->where('created_at', '>=', now()->startOfWeek());
        } elseif ($period === 'month') {
            $query->where('created_at', '>=', now()->startOfMonth());
        }

        $stats = $query->selectRaw('
            COUNT(*) as total_orders,
            SUM(total_amount) as gross_sales,
            SUM(tax_amount) as total_tax,
            SUM(discount_amount) as total_discount,
            SUM(refunded_amount) as total_refunds
        ')->first();

        // Payment Method Breakdown
        $payments = PosPayment::whereHas('order', function($q) use ($business, $period) {
            $q->where('business_id', $business->id);
            if ($period === 'today') $q->whereDate('created_at', today());
        })
        ->select('payment_method', DB::raw('SUM(amount) as total_amount'))
        ->groupBy('payment_method')
        ->get();

        // Cashier Performance
        $cashiers = PosOrder::where('business_id', $business->id)
            ->whereDate('created_at', today())
            ->with('cashier:id,name')
            ->select('cashier_user_id', DB::raw('SUM(total_amount) as sales'), DB::raw('COUNT(*) as count'))
            ->groupBy('cashier_user_id')
            ->get();

        return response()->json([
            'summary' => [
                'orders' => $stats->total_orders ?? 0,
                'gross' => (float)($stats->gross_sales ?? 0),
                'net' => (float)(($stats->gross_sales ?? 0) - ($stats->total_refunds ?? 0)),
                'tax' => (float)($stats->total_tax ?? 0),
                'discounts' => (float)($stats->total_discount ?? 0),
                'refunds' => (float)($stats->total_refunds ?? 0),
            ],
            'payments' => $payments,
            'cashiers' => $cashiers,
        ]);
    }

    /**
     * Get detailed reports and insights.
     */
    public function reports(Request $request, Business $business)
    {
        if (!$business->isMember($request->user())) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        if (!$startDate) {
            $startDate = now()->startOfMonth();
        } else {
            $startDate = \Carbon\Carbon::parse($startDate)->startOfDay();
        }

        if (!$endDate) {
            $endDate = now()->endOfDay();
        } else {
            $endDate = \Carbon\Carbon::parse($endDate)->endOfDay();
        }

        // 1. Sales Summary
        $salesStats = PosOrder::where('business_id', $business->id)
            ->whereBetween('created_at', [$startDate, $endDate])
            ->selectRaw('
                COUNT(*) as total_orders,
                SUM(total_amount) as gross_sales,
                SUM(tax_amount) as total_tax,
                SUM(discount_amount) as total_discount,
                SUM(refunded_amount) as total_refunds
            ')->first();

        // 2. Expenses Summary
        $expenseTotal = \App\Models\PosExpense::where('business_id', $business->id)
            ->whereBetween('expense_date', [$startDate, $endDate])
            ->sum('amount');

        // 3. Sales By Customer
        $customerSales = PosOrder::where('business_id', $business->id)
            ->whereBetween('created_at', [$startDate, $endDate])
            ->whereNotNull('pos_customer_id')
            ->with('customer:id,name')
            ->select('pos_customer_id', DB::raw('SUM(total_amount) as total_spend'), DB::raw('COUNT(*) as order_count'))
            ->groupBy('pos_customer_id')
            ->orderByDesc('total_spend')
            ->limit(10)
            ->get();

        // 4. Payment Methods Breakdown
        $payments = DB::table('pos_payments')
            ->join('pos_orders', 'pos_payments.pos_order_id', '=', 'pos_orders.id')
            ->where('pos_orders.business_id', $business->id)
            ->whereBetween('pos_orders.created_at', [$startDate, $endDate])
            ->select('pos_payments.payment_method', DB::raw('SUM(pos_payments.amount) as total_amount'))
            ->groupBy('pos_payments.payment_method')
            ->get();

        // 5. Top Products & Profitability
        // Calculate COGS using FIFO Batch logs for perfect accuracy
        $cogsData = DB::table('inventory_logs')
            ->leftJoin('inventory_batches', 'inventory_logs.batch_id', '=', 'inventory_batches.id')
            ->leftJoin('listings', 'inventory_logs.listing_id', '=', 'listings.id')
            ->where('inventory_logs.business_id', $business->id)
            ->whereIn('inventory_logs.reason', ['sale', 'sale_pos', 'sale_pos_legacy'])
            ->whereBetween('inventory_logs.created_at', [$startDate, $endDate])
            ->selectRaw('
                SUM(ABS(inventory_logs.change_amount) * COALESCE(inventory_batches.cost_price, listings.purchase_price_amount, 0)) as total_cogs
            ')->first();

        $totalCogs = (float)($cogsData->total_cogs ?? 0);

        // Top Products still derived from orders for better metadata
        $orders = PosOrder::where('business_id', $business->id)
            ->whereBetween('created_at', [$startDate, $endDate])
            ->select('items')
            ->get();

        $productSales = [];
        foreach ($orders as $order) {
            $items = is_array($order->items) ? $order->items : json_decode($order->items, true);
            if (!$items) continue;
            foreach ($items as $item) {
                $id = $item['id'] ?? 'unknown';
                $name = $item['title'] ?? 'Unknown Product';
                $qty = (int)($item['qty'] ?? 1);
                $price = (float)($item['price'] ?? 0);
                $total = (float)($item['total'] ?? ($price * $qty));

                if (!isset($productSales[$id])) {
                    $productSales[$id] = ['id' => $id, 'title' => $name, 'qty' => 0, 'total' => 0];
                }
                $productSales[$id]['qty'] += $qty;
                $productSales[$id]['total'] += $total;
            }
        }

        usort($productSales, fn($a, $b) => $b['total'] <=> $a['total']);
        $topProducts = array_values(array_slice($productSales, 0, 10));

        $netSales = ($salesStats->gross_sales ?? 0) - ($salesStats->total_refunds ?? 0);
        $grossProfit = $netSales - $totalCogs;
        $netProfit = $grossProfit - $expenseTotal;

        return response()->json([
            'summary' => [
                'orders' => $salesStats->total_orders ?? 0,
                'gross' => (float)($salesStats->gross_sales ?? 0),
                'net' => (float)$netSales,
                'tax' => (float)($salesStats->total_tax ?? 0),
                'discounts' => (float)($salesStats->total_discount ?? 0),
                'refunds' => (float)($salesStats->total_refunds ?? 0),
                'expenses' => (float)$expenseTotal,
                'cogs' => (float)$totalCogs,
                'gross_profit' => (float)$grossProfit,
                'net_profit' => (float)$netProfit,
            ],
            'customer_sales' => $customerSales,
            'payments' => $payments,
            'top_products' => $topProducts,
            'period' => [
                'start' => $startDate->toIso8601String(),
                'end' => $endDate->toIso8601String(),
            ]
        ]);
    }
}
