<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\Business;
use App\Models\PosCustomer;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class PosCustomerController extends Controller
{
    /**
     * List/Sync customers for a business.
     */
    public function index(Request $request, Business $business)
    {
        if (!$business->isMember($request->user())) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $lastSync = $request->input('last_sync_timestamp');
        $date = $lastSync ? Carbon::createFromTimestamp($lastSync) : Carbon::create(2000, 1, 1);

        $customers = PosCustomer::where('business_id', $business->id)
            ->where(function($q) use ($date) {
                $q->where('updated_at', '>', $date)
                  ->orWhere('created_at', '>', $date);
            })
            ->withTrashed()
            ->get();

        return response()->json([
            'customers' => $customers,
            'timestamp' => now()->timestamp,
        ]);
    }

    /**
     * Create or update a customer from the POS.
     */
    public function store(Request $request, Business $business)
    {
        if (!$business->isMember($request->user())) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'address' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // Check if customer exists by phone within the business
        $customer = null;
        if ($request->phone) {
            $customer = PosCustomer::where('business_id', $business->id)
                ->where('phone', $request->phone)
                ->first();
        }

        if ($customer) {
            $customer->update($request->only(['name', 'email', 'address', 'notes']));
        } else {
            $customer = PosCustomer::create([
                'business_id' => $business->id,
                'name' => $request->name,
                'phone' => $request->phone,
                'email' => $request->email,
                'address' => $request->address,
                'notes' => $request->notes,
            ]);
        }

        return response()->json([
            'customer' => $customer,
            'status' => 'success'
        ]);
    }

    /**
     * Record a manual payment from a customer.
     */
    public function recordPayment(Request $request, Business $business, PosCustomer $customer)
    {
        if (!$business->isMember($request->user())) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'payment_method' => 'required|string|in:cash,card,mobile_money',
            'reference' => 'nullable|string|max:255',
            'pos_order_id' => 'nullable|integer|exists:pos_orders,id',
        ]);

        $amount = (float)$request->amount;

        \Illuminate\Support\Facades\DB::beginTransaction();
        try {
            // Update customer balance
            $customer->decrement('balance', $amount);

            // Record in ledger
            $customer->ledger()->create([
                'business_id' => $business->id,
                'pos_order_id' => $request->pos_order_id,
                'type' => 'credit', // Credit means money coming in/reducing debt
                'payment_method' => $request->payment_method,
                'amount' => $amount,
                'notes' => 'Manual payment recorded' . ($request->reference ? ' - Ref: ' . $request->reference : ''),
            ]);

            \Illuminate\Support\Facades\DB::commit();

            return response()->json([
                'status' => 'success',
                'balance' => $customer->balance,
                'message' => 'Payment recorded successfully',
            ]);
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\DB::rollBack();
            return response()->json(['message' => 'Failed to record payment', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Record a manual transaction (debit or credit).
     */
    public function recordTransaction(Request $request, Business $business, PosCustomer $customer)
    {
        if (!$business->isMember($request->user())) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'type' => 'required|in:debit,credit', // debit = increases debt, credit = increases deposit/reduces debt
            'notes' => 'required|string|max:255',
            'reference' => 'nullable|string|max:255',
        ]);

        $amount = (float)$request->amount;
        $type = $request->type;

        \Illuminate\Support\Facades\DB::beginTransaction();
        try {
            if ($type === 'debit') {
                $customer->increment('balance', $amount);
            } else {
                $customer->decrement('balance', $amount);
            }

            // Record in ledger
            $customer->ledger()->create([
                'business_id' => $business->id,
                'type' => $type,
                'amount' => $amount,
                'notes' => $request->notes . ($request->reference ? ' - Ref: ' . $request->reference : ''),
            ]);

            \Illuminate\Support\Facades\DB::commit();

            return response()->json([
                'status' => 'success',
                'balance' => $customer->balance,
                'message' => 'Transaction recorded successfully',
            ]);
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\DB::rollBack();
            return response()->json(['message' => 'Failed to record transaction', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Get customer transaction ledger history.
     */
    public function ledger(Request $request, Business $business, PosCustomer $customer)
    {
        if (!$business->isMember($request->user())) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        // 1. Get all orders for this customer (full purchase history)
        // We eager load 'ledger' entries that are linked to these orders
        $orders = $customer->orders()
            ->with(['payments', 'ledger'])
            ->orderBy('created_at', 'desc')
            ->get();

        // 2. Get manual payments/transactions (ledger entries without an order_id OR not already accounted for)
        // Actually, we want ledger entries that are NOT linked to ANY order, 
        // OR ledger entries that ARE linked but we want to show them as separate tiles if they are manual.
        // Wait, if they are linked to an order, they should contribute to THAT order's balance.
        // If they are manual payments, they should also show up as tiles in the ledger list.
        $manualEntries = $customer->ledger()
            ->whereNull('pos_order_id')
            ->orderBy('created_at', 'desc')
            ->get();

        $combined = [];

        // Add orders as 'debit' (Sales)
        foreach ($orders as $order) {
            $combined[] = [
                'type' => 'debit',
                'amount' => $order->total_amount,
                'notes' => 'Sale Purchase - Order #' . ($order->receipt_number ?? $order->id),
                'created_at' => $order->created_at,
                'order' => [
                    'id' => $order->id,
                    'receipt_number' => $order->receipt_number,
                    'total_amount' => $order->total_amount,
                    'status' => $order->status,
                    'refunded_amount' => $order->refunded_amount,
                    'payments' => $order->payments,
                    'ledger_payments' => $order->ledger->where('type', 'credit')->values(), // Manual payments linked to this order
                    'items' => $order->items,
                    'offline_uuid' => $order->offline_uuid,
                ],
            ];
        }

        // Add manual/unlinked ledger entries
        foreach ($manualEntries as $entry) {
            $combined[] = [
                'id' => $entry->id,
                'type' => $entry->type,
                'payment_method' => $entry->payment_method,
                'amount' => $entry->amount,
                'notes' => $entry->notes,
                'created_at' => $entry->created_at,
                'order' => null,
            ];
        }

        // Also add LINKED manual payments to the main list so they appear as tiles
        // but only if they are 'credit' (payments)
        $linkedManualPayments = $customer->ledger()
            ->whereNotNull('pos_order_id')
            ->where('type', 'credit')
            ->get();

        foreach ($linkedManualPayments as $entry) {
            $combined[] = [
                'id' => $entry->id,
                'type' => $entry->type,
                'payment_method' => $entry->payment_method,
                'amount' => $entry->amount,
                'notes' => $entry->notes,
                'created_at' => $entry->created_at,
                'order' => [
                    'id' => $entry->order->id,
                    'receipt_number' => $entry->order->receipt_number,
                ],
            ];
        }

        // Sort combined list by created_at (most recent first)
        usort($combined, function ($a, $b) {
            return $b['created_at'] <=> $a['created_at'];
        });

        return response()->json([
            'ledger' => $combined,
            'status' => 'success'
        ]);
    }
}
