<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\UserSubscription;
use App\Notifications\SubscriptionActivatedNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class RelworxWebhookController extends Controller
{
    /**
     * Handle incoming Relworx webhook requests.
     *
     * @param  Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function handle(Request $request)
    {
        Log::info('Relworx Webhook Received:', $request->all());

        if (!$this->verifySignature($request)) {
            Log::warning('Relworx Webhook Signature Verification Failed');
            return response()->json(['success' => false, 'message' => 'Invalid signature'], 401);
        }

        $status = $request->input('status');
        $customerReference = $request->input('customer_reference');
        $internalReference = $request->input('internal_reference');

        $subscription = UserSubscription::where('transaction_id', $customerReference)->first();

        if (!$subscription) {
            Log::error("Relworx Webhook: Subscription not found for reference {$customerReference}");
            return response()->json(['success' => false, 'message' => 'Subscription not found'], 404);
        }

        if ($status === 'success') {
            $package = $subscription->package;
            $businessId = $subscription->business_id;

            $startsAt = now();
            $expiresAt = now()->addDays($package->duration_days);

            // Recovery Logic: If subscription has no business_id, find the owner's first business
            if (!$businessId && $subscription->user_id) {
                $firstBusiness = \App\Models\Business::where('owner_user_id', $subscription->user_id)->first();
                if ($firstBusiness) {
                    $businessId = $firstBusiness->id;
                    $subscription->update(['business_id' => $businessId]);
                    Log::info("Relworx Webhook: Recovered business_id {$businessId} for orphaned subscription {$subscription->id}");
                }
            }

            if ($businessId) {
                // Check for current active subscription
                $currentActive = UserSubscription::where('business_id', $businessId)
                    ->where('status', 'active')
                    ->where('expires_at', '>', now())
                    ->first();

                if ($currentActive) {
                    $currentPackage = $currentActive->package;

                    if ($currentPackage->id == $package->id) {
                        // RENEWAL: Add days to current expiry
                        $startsAt = $currentActive->expires_at;
                        $expiresAt = $startsAt->copy()->addDays($package->duration_days);
                        
                        // We keep both active? No, usually business logic prefers one active at a time 
                        // but since Business->activeSubscription() uses latest('expires_at'), it works.
                        // However, let's mark the old one as 'completed' or just let it coexist.
                        // User request: "add the days remaining to the next 30 days"
                        // This effectively means the NEW one starts when the OLD one ends.
                    } elseif ($package->price > $currentPackage->price) {
                        // UPGRADE: Start immediately, cancel old
                        $currentActive->update(['status' => 'cancelled', 'notes' => 'Upgraded to ' . $package->name]);
                        $startsAt = now();
                        $expiresAt = now()->addDays($package->duration_days);
                    } else {
                        // DOWNGRADE: Schedule for later
                        $startsAt = $currentActive->expires_at;
                        $expiresAt = $startsAt->copy()->addDays($package->duration_days);
                    }
                }
            }
            
            $subscription->update([
                'status' => 'active',
                'starts_at' => $startsAt,
                'expires_at' => $expiresAt,
                'notes' => ($subscription->notes ? $subscription->notes . "\n" : "") . "Relworx Completed: {$internalReference}",
            ]);

            Log::info("Relworx Webhook: Subscription activated/scheduled for business {$businessId}");

            // Update all existing published listings to point to this new subscription
            // This ensures they enjoy perks like Auto-Bump immediately.
            // Using a query instead of models to avoid triggering re-review observers.
            if ($status === 'success' && $startsAt <= now()) {
                $maxListings = (int) $package->max_listings;
                $rank = (int) $package->sort_order;

                $listingIds = \App\Models\Listing::where('business_id', $businessId)
                    ->where('status', 'published')
                    ->orderByDesc('bumped_at')
                    ->orderByDesc('published_at')
                    ->pluck('id')
                    ->toArray();

                $toKeep = array_slice($listingIds, 0, $maxListings);
                $toDraft = array_slice($listingIds, $maxListings);

                if (!empty($toKeep)) {
                    \App\Models\Listing::whereIn('id', $toKeep)
                        ->update([
                            'subscription_id' => $subscription->id,
                            'promoted_rank' => $rank
                        ]);
                }

                if (!empty($toDraft)) {
                    \App\Models\Listing::whereIn('id', $toDraft)
                        ->update(['status' => 'draft']);
                }
                    
                Log::info("Relworx Webhook: Balanced listings for business {$businessId} (Kept: " . count($toKeep) . ", Drafted: " . count($toDraft) . ")");
            }

            // Notify User
            try {
                $userToNotify = $subscription->user;
                if (!$userToNotify && $subscription->business && $subscription->business->owner) {
                    $userToNotify = $subscription->business->owner;
                }

                if ($userToNotify) {
                    $userToNotify->notify(new SubscriptionActivatedNotification($subscription));
                    Log::info("Relworx Webhook: Sent activation notification to User ID {$userToNotify->id}");
                }
            } catch (\Throwable $e) {
                Log::error("Relworx Webhook: Failed to send notification: " . $e->getMessage());
            }
        } elseif ($status === 'failed') {
            $subscription->update(['status' => 'failed']);
            Log::warning("Relworx Webhook: Payment failed for reference {$customerReference}");
        }

        return response()->json(['success' => true]);
    }

    /**
     * Verify the Relworx webhook signature.
     *
     * @param  Request  $request
     * @return bool
     */
    protected function verifySignature(Request $request)
    {
        $signatureHeader = $request->header('Relworx-Signature');
        if (!$signatureHeader) {
            return false;
        }

        // Parse signature header: t=1561370460,v=fgrSxEFI/z6Twr6xZogRYnKCfew=
        $parts = explode(',', $signatureHeader);
        $timestamp = null;
        $providedSignature = null;

        foreach ($parts as $part) {
            $item = explode('=', $part);
            if (count($item) !== 2) continue;
            
            if ($item[0] === 't') $timestamp = $item[1];
            if ($item[0] === 'v') $providedSignature = $item[1];
        }

        if (!$timestamp || !$providedSignature) {
            return false;
        }

        // Check timestamp against current time (max 5 minutes difference)
        if (abs(time() - $timestamp) > 300) {
            Log::warning('Relworx Webhook: Signature timestamp expired');
            return false;
        }

        $webhookUrl = $request->fullUrl();
        $secret = config('services.relworx.webhook_secret');

        // Create base string: URL + timestamp
        $baseString = $webhookUrl . $timestamp;

        // Sort POST variables alphabetically and append
        $data = $request->only(['status', 'customer_reference', 'internal_reference']);
        ksort($data);
        
        foreach ($data as $key => $value) {
            $baseString .= $key . $value;
        }

        // Hash using HMAC-SHA256
        $computedSignature = hash_hmac('sha256', $baseString, $secret);

        // Convert base64 to hex if provided signature is base64 (Relworx docs show both. Let's try matching hex)
        // Docs say "generate a hex-encoded signature"
        return hash_equals($computedSignature, $providedSignature);
    }
}
