<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Requests\Listing\ReverseImageSearchRequest;
use App\Http\Resources\ApiResponse;
use App\Http\Resources\ListingResource;
use App\Services\ReverseImageSearchService;
use App\Services\SearchTrackingService;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Storage;

class ReverseImageSearchController extends Controller
{
    protected ReverseImageSearchService $searchService;
    protected SearchTrackingService $searchTracker;

    public function __construct(
        ReverseImageSearchService $searchService,
        SearchTrackingService $searchTracker
    ) {
        $this->searchService = $searchService;
        $this->searchTracker = $searchTracker;
    }

    /**
     * Search for similar listings by uploading an image
     * Combines content hash matching and visual similarity detection
     */
    public function search(ReverseImageSearchRequest $request): JsonResponse
    {
        $file = $request->file('image');
        $limit = (int) $request->input('limit', 20);

        // Store temporary image for analysis
        $tempPath = $file->store('temp-search', 'public');

        try {
            // Perform reverse image search
            $results = $this->searchService->searchByImage($tempPath, $limit);

            // Format results
            $formattedResults = $results->map(function ($item) {
                return [
                    'listing' => new ListingResource($item['listing']),
                    'similarity_score' => $item['similarity_score'],
                    'match_type' => $item['match_type'],
                ];
            });

            $this->searchTracker->trackSearch(
                $request,
                'image',
                null,
                ['limit' => $limit],
                $results->count()
            );

            return ApiResponse::success([
                'results' => $formattedResults,
                'total_found' => $results->count(),
                'search_method' => 'hybrid', // hash + visual
            ]);

        } finally {
            // Clean up temporary file
            if (Storage::disk('public')->exists($tempPath)) {
                Storage::disk('public')->delete($tempPath);
            }
        }
    }
}
