<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Services\SearchTrackingService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class SearchAnalyticsController extends Controller
{
    protected SearchTrackingService $searchTrackingService;

    public function __construct(SearchTrackingService $searchTrackingService)
    {
        $this->searchTrackingService = $searchTrackingService;
    }

    public function popularSearches(Request $request): JsonResponse
    {
        $request->validate([
            'limit' => 'sometimes|integer|min:1|max:100',
            'search_type' => 'sometimes|string|in:text,image,category,location',
        ]);

        $popularSearches = $this->searchTrackingService->getPopularSearches(
            $request->input('limit', 10),
            $request->input('search_type')
        );

        return response()->json([
            'success' => true,
            'data' => $popularSearches,
        ]);
    }

    public function userHistory(Request $request): JsonResponse
    {
        $userId = auth('api')->id();

        if (!$userId) {
            return response()->json([
                'success' => false,
                'message' => 'Authentication required',
            ], 401);
        }

        $request->validate([
            'limit' => 'sometimes|integer|min:1|max:100',
        ]);

        $history = $this->searchTrackingService->getUserSearchHistory(
            $userId,
            $request->input('limit', 50)
        );

        return response()->json([
            'success' => true,
            'data' => $history,
        ]);
    }

    public function analytics(Request $request): JsonResponse
    {
        $request->validate([
            'search_type' => 'sometimes|string|in:text,image,category,location',
            'period' => 'sometimes|string|in:24hours,7days,30days,90days',
        ]);

        $analytics = $this->searchTrackingService->getSearchAnalytics(
            $request->input('search_type'),
            $request->input('period', '7days')
        );

        return response()->json([
            'success' => true,
            'data' => $analytics,
        ]);
    }

    public function zeroResultSearches(Request $request): JsonResponse
    {
        $request->validate([
            'limit' => 'sometimes|integer|min:1|max:100',
            'search_type' => 'sometimes|string|in:text,image,category,location',
        ]);

        $zeroResults = $this->searchTrackingService->getZeroResultSearches(
            $request->input('limit', 20),
            $request->input('search_type')
        );

        return response()->json([
            'success' => true,
            'data' => $zeroResults,
            'message' => 'Searches that returned no results - useful for improving search functionality',
        ]);
    }
}
