<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Resources\ApiResponse;
use App\Models\PremiumPackage;
use App\Models\UserSubscription;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class SubscriptionController extends Controller
{
    public function subscribe(Request $request): JsonResponse
    {
        $request->validate([
            'package_id' => 'required|integer|exists:premium_packages,id',
            'payment_method' => 'required|string',
            'transaction_id' => 'required|string',
            'business_id' => 'nullable|integer|exists:businesses,id',
        ]);

        $userId = auth('api')->id();
        $businessId = $request->input('business_id');
        $package = PremiumPackage::findOrFail($request->input('package_id'));

        if (!$package->is_active) {
            return ApiResponse::error('This package is not available', 400);
        }

        $activeSubscription = UserSubscription::where('user_id', $userId)
            ->where('status', 'active')
            ->where('expires_at', '>', now())
            ->first();

        if ($activeSubscription) {
            return ApiResponse::error('You already have an active subscription', 400);
        }

        $subscription = UserSubscription::create([
            'user_id' => $userId,
            'business_id' => $businessId,
            'package_id' => $package->id,
            'status' => 'active',
            'starts_at' => now(),
            'expires_at' => now()->addDays($package->duration_days),
            'amount_paid' => $package->price,
            'payment_method' => $request->input('payment_method'),
            'transaction_id' => $request->input('transaction_id'),
        ]);

        $subscription->load('package');

        return ApiResponse::success([
            'message' => 'Subscription activated successfully',
            'subscription' => $subscription,
        ], 201);
    }

    public function mySubscriptions(Request $request): JsonResponse
    {
        $userId = auth('api')->id();
        $businessId = $request->input('business_id');

        $query = UserSubscription::with('package')->latest();

        if ($businessId) {
            $query->where('business_id', $businessId);
        } else {
            $query->where('user_id', $userId);
        }

        $subscriptions = $query->get();

        $activeSubscription = $subscriptions->firstWhere('status', 'active');

        return ApiResponse::success([
            'subscriptions' => $subscriptions,
            'active_subscription' => $activeSubscription,
        ]);
    }

    public function currentSubscription(Request $request): JsonResponse
    {
        $userId = auth('api')->id();
        $businessId = $request->input('business_id');

        $query = UserSubscription::where('status', 'active')
            ->where('expires_at', '>', now())
            ->with('package');

        if ($businessId) {
            $query->where('business_id', $businessId);
        } else {
            $query->where('user_id', $userId);
        }

        $subscription = $query->first();

        if (!$subscription) {
            return ApiResponse::error('No active subscription found', 404);
        }

        return ApiResponse::success([
            'subscription' => $subscription,
            'business_id' => $subscription->business_id,
            'usage' => [
                'listings_used' => $subscription->listings_used,
                'listings_remaining' => $subscription->package->max_listings - $subscription->listings_used,
                'featured_used' => $subscription->featured_listings_used,
                'featured_remaining' => $subscription->package->max_featured_listings - $subscription->featured_listings_used,
                'bumps_used_this_month' => $subscription->bumps_used_this_month,
                'bumps_remaining' => $subscription->package->max_bumps_per_month - $subscription->bumps_used_this_month,
            ],
        ]);
    }

    public function cancel(int $subscriptionId): JsonResponse
    {
        $userId = auth('api')->id();

        $subscription = UserSubscription::where('id', $subscriptionId)
            ->where('user_id', $userId)
            ->first();

        if (!$subscription) {
            return ApiResponse::error('Subscription not found', 404);
        }

        if ($subscription->status === 'cancelled') {
            return ApiResponse::error('Subscription is already cancelled', 400);
        }

        $subscription->update([
            'status' => 'cancelled',
            'cancelled_at' => now(),
        ]);

        return ApiResponse::success([
            'message' => 'Subscription cancelled successfully. You can continue using premium features until the expiry date.',
            'subscription' => $subscription,
        ]);
    }

    public function checkEligibility(): JsonResponse
    {
        $userId = auth('api')->id();

        $activeSubscription = UserSubscription::where('user_id', $userId)
            ->where('status', 'active')
            ->where('expires_at', '>', now())
            ->with('package')
            ->first();

        if (!$activeSubscription) {
            return ApiResponse::success([
                'has_subscription' => false,
                'can_subscribe' => true,
                'message' => 'You can subscribe to a premium package',
            ]);
        }

        return ApiResponse::success([
            'has_subscription' => true,
            'can_subscribe' => false,
            'subscription' => $activeSubscription,
            'can_create_listing' => $activeSubscription->canCreateListing(),
            'can_feature_listing' => $activeSubscription->canFeatureListing(),
            'can_bump_listing' => $activeSubscription->canBumpListing(),
        ]);
    }
}
