<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Requests\User\ChangePasswordRequest;
use App\Http\Requests\User\UpdateProfileRequest;
use App\Http\Resources\ApiResponse;
use App\Http\Resources\UserResource;
use App\Models\JwtRefreshToken;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class UserController extends Controller
{
    /**
     * Get the authenticated user's profile.
     */
    public function profile(): JsonResponse
    {
        return ApiResponse::success(new UserResource(auth('api')->user()));
    }

    /**
     * Update the authenticated user's profile.
     */
    public function updateProfile(UpdateProfileRequest $request): JsonResponse
    {
        $user = auth('api')->user();
        $data = $request->validated();

        if ($request->hasFile('profile_photo')) {
            if ($user->profile_photo && !filter_var($user->profile_photo, FILTER_VALIDATE_URL)) {
                Storage::disk('public')->delete($user->profile_photo);
            }

            $path = $request->file('profile_photo')->store('profile-photos', 'public');
            $data['profile_photo'] = $path;
        }

        if (isset($data['phone']) && $data['phone'] !== $user->phone) {
            $data['phone_verified_at'] = null;
            $data['is_phone_verified'] = false;
        }

        if (isset($data['secondary_phone']) && $data['secondary_phone'] !== $user->secondary_phone) {
            $data['secondary_phone_verified_at'] = null;
        }

        if (isset($data['whatsapp_number']) && $data['whatsapp_number'] !== $user->whatsapp_number) {
            $data['whatsapp_verified_at'] = null;
        }

        if (isset($data['secondary_whatsapp_number']) && $data['secondary_whatsapp_number'] !== $user->secondary_whatsapp_number) {
            $data['secondary_whatsapp_verified_at'] = null;
        }

        $user->update($data);

        return ApiResponse::success(
            new UserResource($user->fresh()),
            'Profile updated successfully.'
        );
    }

    /**
     * Change the authenticated user's password.
     */
    public function changePassword(ChangePasswordRequest $request): JsonResponse
    {
        $user = auth('api')->user();

        $user->update([
            'password' => $request->password,
        ]);

        JwtRefreshToken::where('user_id', $user->id)
            ->where('id', '!=', $this->getCurrentRefreshTokenId())
            ->update(['revoked_at' => now()]);

        return ApiResponse::success(null, 'Password changed successfully. Other sessions have been logged out.');
    }

    /**
     * Soft delete the authenticated user's account.
     */
    public function deleteAccount(Request $request): JsonResponse
    {
        $user = auth('api')->user();

        $request->validate([
            'password' => ['required', 'string', 'current_password:api'],
            'reason' => ['nullable', 'string', 'max:500'],
        ]);

        $timestamp = time();
        $user->update([
            'status' => User::STATUS_DELETED,
            'email' => substr("deleted_{$timestamp}_{$user->email}", 0, 255),
            'phone' => $user->phone ? substr("del_{$timestamp}_{$user->phone}", 0, 20) : null,
            'username' => $user->username ? substr("del_{$timestamp}_{$user->username}", 0, 255) : null,
            'referral_code' => substr("del_{$timestamp}_{$user->referral_code}", 0, 20),
        ]);

        JwtRefreshToken::where('user_id', $user->id)->update(['revoked_at' => now()]);

        $user->delete();

        auth('api')->logout();

        return ApiResponse::success(null, 'Account deleted successfully.');
    }

    /**
     * Remove profile photo.
     */
    public function removeProfilePhoto(): JsonResponse
    {
        $user = auth('api')->user();

        if ($user->profile_photo && !filter_var($user->profile_photo, FILTER_VALIDATE_URL)) {
            Storage::disk('public')->delete($user->profile_photo);
        }

        $user->update(['profile_photo' => null]);

        return ApiResponse::success(
            new UserResource($user->fresh()),
            'Profile photo removed successfully.'
        );
    }

    /**
     * Get the current refresh token ID.
     */
    private function getCurrentRefreshTokenId(): ?int
    {
        try {
            $payload = auth('api')->payload();
            $jti = $payload->get('jti');

            $refreshToken = JwtRefreshToken::where('token_id', $jti)->first();

            return $refreshToken?->id;
        } catch (\Exception $e) {
            return null;
        }
    }
}
