<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Resources\ApiResponse;
use App\Http\Resources\ListingResource;
use App\Models\Listing;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class UserListingController extends Controller
{
    /**
     * List all listings for the authenticated user across all businesses they own or are members of.
     */
    public function index(Request $request): JsonResponse
    {
        $user = auth('api')->user();

        $perPage = (int) $request->query('per_page', 20);
        $perPage = max(1, min(100, $perPage));

        // Aggregate listings from all businesses the user has management access to
        $query = Listing::query()
            ->whereHas('business', function ($q) use ($user) {
                $q->where('owner_user_id', $user->id)
                    ->orWhereHas('memberships', function ($mq) use ($user) {
                        $mq->where('user_id', $user->id);
                    });
            });

        // Filter by status if provided (e.g., 'published', 'review', 'rejected', 'sold', 'expired')
        if ($request->filled('status')) {
            $status = $request->query('status');
            
            // Map common requested statuses to actual model constants if needed
            if ($status === 'declined') {
                $status = Listing::STATUS_REJECTED;
            } elseif ($status === 'closed') {
                $status = Listing::STATUS_SOLD;
            }
            
            $query->where('status', $status);
        }

        // Filter by specific business if provided
        if ($request->filled('business_id')) {
            $query->where('business_id', $request->query('business_id'));
        }

        // Search within user listings
        if ($request->filled('q')) {
            $q = $request->query('q');
            $query->where(function ($sub) use ($q) {
                $sub->where('title', 'like', "%{$q}%")
                    ->orWhere('description', 'like', "%{$q}%");
            });
        }

        $listings = $query
            ->with(['media', 'business'])
            ->orderByDesc('created_at')
            ->paginate($perPage);

        return ApiResponse::success([
            'listings' => ListingResource::collection($listings->items()),
            'pagination' => [
                'current_page' => $listings->currentPage(),
                'per_page' => $listings->perPage(),
                'total' => $listings->total(),
                'last_page' => $listings->lastPage(),
            ],
        ]);
    }
}
