<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Models\Business;
use App\Models\BusinessInvitation;
use App\Models\BusinessMembership;
use App\Models\User;
use App\Mail\BusinessInvitationEmail;
use App\Mail\BusinessInvitationStatusEmail;
use App\Mail\BusinessOwnershipTransferredOldOwnerEmail;
use App\Mail\BusinessOwnershipTransferredNewOwnerEmail;
use App\Mail\BusinessInvitationRevokedEmail;
use App\Mail\BusinessMemberRemovedEmail;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;

class BusinessController extends Controller
{
    public function __construct(protected \App\Services\FcmService $fcmService)
    {
    }

    public function edit(): View
    {
        $businessId = session('selected_business_id');
        
        if (!$businessId) {
            abort(404, 'No business selected');
        }

        $business = Business::where('id', $businessId)
            ->where(function ($query) {
                $query->where('owner_user_id', Auth::guard('web')->id())
                    ->orWhereHas('memberships', function ($q) {
                        $q->where('user_id', Auth::guard('web')->id());
                    });
            })
            ->firstOrFail();

        return view('web.business.edit', compact('business'));
    }

    public function update(Request $request): RedirectResponse
    {
        $businessId = session('selected_business_id');
        
        if (!$businessId) {
            return back()->withErrors(['error' => 'No business selected']);
        }

        $business = Business::where('id', $businessId)
            ->where(function ($query) {
                $query->where('owner_user_id', Auth::guard('web')->id())
                    ->orWhereHas('memberships', function ($q) {
                        $q->where('user_id', Auth::guard('web')->id());
                    });
            })
            ->firstOrFail();

        $validated = $request->validate([
            'name' => 'required|string|min:2|max:255',
            'description' => 'required|string|min:10',
            'business_type' => 'required|string|in:individual,retail,wholesale,serviceProvider',
            'business_category' => 'nullable|string',
            'address' => 'required|string|min:5',
            'district' => 'required|string',
        ]);

        $business->update($validated);

        return back()->with('success', 'Business details updated successfully.');
    }

    public function links(): View
    {
        $businessId = session('selected_business_id');
        
        if (!$businessId) {
            abort(404, 'No business selected');
        }

        $business = Business::where('id', $businessId)
            ->where(function ($query) {
                $query->where('owner_user_id', Auth::guard('web')->id())
                    ->orWhereHas('memberships', function ($q) {
                        $q->where('user_id', Auth::guard('web')->id());
                    });
            })
            ->firstOrFail();

        return view('web.business.links', compact('business'));
    }

    public function updateLinks(Request $request): RedirectResponse
    {
        $businessId = session('selected_business_id');
        
        if (!$businessId) {
            return back()->withErrors(['error' => 'No business selected']);
        }

        $business = Business::where('id', $businessId)
            ->where(function ($query) {
                $query->where('owner_user_id', Auth::guard('web')->id())
                    ->orWhereHas('memberships', function ($q) {
                        $q->where('user_id', Auth::guard('web')->id());
                    });
            })
            ->firstOrFail();

        $validated = $request->validate([
            'website' => 'nullable|url|max:255',
            'facebook_url' => 'nullable|url|max:255',
            'twitter_url' => 'nullable|url|max:255',
            'instagram_url' => 'nullable|url|max:255',
            'linkedin_url' => 'nullable|url|max:255',
            'tiktok_url' => 'nullable|url|max:255',
            'youtube_url' => 'nullable|url|max:255',
        ]);

        $business->update($validated);

        return back()->with('success', 'Business links updated successfully.');
    }

    public function branding(): View
    {
        $businessId = session('selected_business_id');
        
        if (!$businessId) {
            abort(404, 'No business selected');
        }

        $business = Business::where('id', $businessId)
            ->where(function ($query) {
                $query->where('owner_user_id', Auth::guard('web')->id())
                    ->orWhereHas('memberships', function ($q) {
                        $q->where('user_id', Auth::guard('web')->id());
                    });
            })
            ->firstOrFail();

        return view('web.business.branding', compact('business'));
    }

    public function updateBranding(Request $request): RedirectResponse
    {
        $businessId = session('selected_business_id');
        
        if (!$businessId) {
            return back()->withErrors(['error' => 'No business selected']);
        }

        $business = Business::where('id', $businessId)
            ->where(function ($query) {
                $query->where('owner_user_id', Auth::guard('web')->id())
                    ->orWhereHas('memberships', function ($q) {
                        $q->where('user_id', Auth::guard('web')->id());
                    });
            })
            ->firstOrFail();

        $validated = $request->validate([
            'logo' => 'nullable|image|max:2048',
            'cover' => 'nullable|image|max:5120',
            'primary_color' => 'nullable|string|regex:/^#[0-9A-Fa-f]{6}$/',
            'secondary_color' => 'nullable|string|regex:/^#[0-9A-Fa-f]{6}$/',
        ]);

        // Handle logo upload
        if ($request->hasFile('logo')) {
            // Delete old logo if exists
            if ($business->logo) {
                Storage::disk('public')->delete($business->logo);
            }
            $validated['logo'] = $request->file('logo')->store('business-logos', 'public');
        }

        // Handle cover upload
        if ($request->hasFile('cover')) {
            // Delete old cover if exists
            if ($business->cover) {
                Storage::disk('public')->delete($business->cover);
            }
            $validated['cover'] = $request->file('cover')->store('business-covers', 'public');
        }

        $business->update($validated);

        return back()->with('success', 'Business branding updated successfully.');
    }

    public function editNumber(string $type): View
    {
        if (!in_array($type, ['phone', 'whatsapp'])) {
            abort(404);
        }

        $businessId = session('selected_business_id');
        
        if (!$businessId) {
            abort(404, 'No business selected');
        }

        $business = Business::where('id', $businessId)
            ->where(function ($query) {
                $query->where('owner_user_id', Auth::guard('web')->id())
                    ->orWhereHas('memberships', function ($q) {
                        $q->where('user_id', Auth::guard('web')->id());
                    });
            })
            ->firstOrFail();

        return view('web.business.edit-number', compact('business', 'type'));
    }

    public function updateNumber(Request $request, string $type): RedirectResponse
    {
        if (!in_array($type, ['phone', 'whatsapp'])) {
            abort(404);
        }

        $businessId = session('selected_business_id');
        
        if (!$businessId) {
            return back()->withErrors(['error' => 'No business selected']);
        }

        $business = Business::where('id', $businessId)
            ->where(function ($query) {
                $query->where('owner_user_id', Auth::guard('web')->id())
                    ->orWhereHas('memberships', function ($q) {
                        $q->where('user_id', Auth::guard('web')->id());
                    });
            })
            ->firstOrFail();

        $field = $type === 'phone' ? 'primary_phone' : 'primary_whatsapp';
        
        $validated = $request->validate([
            'number' => [
                'required',
                'string',
                'regex:/^\+256[0-9]{9}$/',
                function ($attribute, $value, $fail) use ($business, $field) {
                    $exists = Business::where($field, $value)
                        ->where('id', '!=', $business->id)
                        ->exists();
                    
                    if ($exists) {
                        $fail('This number is already in use by another business.');
                    }
                },
            ],
        ], [
            'number.required' => 'Number is required.',
            'number.regex' => 'Please enter a valid Ugandan phone number (9 digits after +256).',
        ]);

        $business->update([
            $field => $validated['number']
        ]);

        $label = $type === 'phone' ? 'Phone number' : 'WhatsApp number';
        return back()->with('success', "$label updated successfully.");
    }

    public function members(): View
    {
        $businessId = session('selected_business_id');
        
        if (!$businessId) {
            abort(404, 'No business selected');
        }

        $business = Business::where('id', $businessId)
            ->where(function ($query) {
                $query->where('owner_user_id', Auth::guard('web')->id())
                    ->orWhereHas('memberships', function ($q) {
                        $q->where('user_id', Auth::guard('web')->id());
                    });
            })
            ->firstOrFail();

        $members = $business->members()
            ->select('users.*')
            ->get()
            ->map(function (User $user) use ($business) {
                $membership = $business->memberships()->where('user_id', $user->id)->first();
                $user->role = $membership?->role;
                $user->joined_at = $membership?->joined_at;
                return $user;
            });
            
        $invitations = BusinessInvitation::where('business_id', $business->id)
            ->whereNull('accepted_at')
            ->whereNull('revoked_at')
            ->where(function ($q) {
                $q->whereNull('expires_at')->orWhere('expires_at', '>', now());
            })
            ->get();

        return view('web.business.members', compact('business', 'members', 'invitations'));
    }

    public function invite(Request $request): RedirectResponse
    {
        $businessId = session('selected_business_id');
        
        if (!$businessId) {
            return back()->withErrors(['error' => 'No business selected']);
        }

        $business = Business::where('id', $businessId)
            ->where(function ($query) {
                $query->where('owner_user_id', Auth::guard('web')->id())
                    ->orWhereHas('memberships', function ($q) {
                        $q->where('user_id', Auth::guard('web')->id());
                    });
            })
            ->firstOrFail();

        if ((int)$business->owner_user_id !== (int)Auth::guard('web')->id()) {
             return back()->withErrors(['error' => 'Only the business owner can invite new members.']);
        }

        $request->validate([
            'email' => 'required|email',
            'role' => 'required|in:manager,editor',
        ]);

        $email = strtolower(trim($request->email));

        if (strtolower(Auth::guard('web')->user()->email) === $email) {
            return back()->withErrors(['email' => 'You cannot invite yourself.']);
        }

        $existingUser = User::where('email', $email)->first();
        if (!$existingUser) {
            return back()->withErrors(['email' => 'User must have an account.']);
        }

        if ($business->memberships()->where('user_id', $existingUser->id)->exists()) {
             return back()->withErrors(['email' => 'User is already a member.']);
        }

        $existingInvite = BusinessInvitation::where('business_id', $business->id)
            ->where('email', $email)
            ->whereNull('accepted_at')
            ->whereNull('revoked_at')
            ->where(function ($q) {
                $q->whereNull('expires_at')->orWhere('expires_at', '>', now());
            })
            ->first();

        if ($existingInvite) {
             return back()->withErrors(['email' => 'Invitation already sent.']);
        }

        $invite = BusinessInvitation::create([
            'business_id' => $business->id,
            'invited_by_user_id' => Auth::guard('web')->id(),
            'email' => $email,
            'role' => $request->role,
            'expires_at' => now()->addDays(7),
        ]);

        // Send Email (ignoring failure to not break flow, similar to API)
        // Send Email & Notification
        try {
            Mail::to($existingUser->email)->send(new BusinessInvitationEmail(
                $business,
                $invite,
                Auth::guard('web')->user(),
                $existingUser
            ));

            $this->fcmService->sendToUser(
                $existingUser->id,
                'Business Invitation',
                Auth::guard('web')->user()->name . " invited you to manage {$business->name}.",
                ['type' => 'business_invitation', 'business_id' => $business->id]
            );
        } catch (\Throwable $e) {
            \Log::warning('Failed to send invite notifications', ['error' => $e->getMessage()]);
        }

        return back()->with('success', 'Invitation sent successfully.');
    }

    public function updateMemberRole(Request $request, $userId): RedirectResponse
    {
        $businessId = session('selected_business_id');
        
        if (!$businessId) {
            return back()->withErrors(['error' => 'No business selected']);
        }

        $business = Business::where('id', $businessId)
            ->where(function ($query) {
                $query->where('owner_user_id', Auth::guard('web')->id())
                    ->orWhereHas('memberships', function ($q) {
                        $q->where('user_id', Auth::guard('web')->id());
                    });
            })
            ->firstOrFail();

        if ((int)$business->owner_user_id === (int)$userId) {
             return back()->withErrors(['error' => 'Cannot change owner role.']);
        }
        
        $request->validate(['role' => 'required|in:manager,editor']);

        BusinessMembership::where('business_id', $business->id)
            ->where('user_id', $userId)
            ->update(['role' => $request->role]);

        return back()->with('success', 'Role updated successfully.');
    }

    public function removeMember($userId): RedirectResponse
    {
        $businessId = session('selected_business_id');
        
        if (!$businessId) {
            return back()->withErrors(['error' => 'No business selected']);
        }

        $business = Business::where('id', $businessId)
            ->where(function ($query) {
                $query->where('owner_user_id', Auth::guard('web')->id())
                    ->orWhereHas('memberships', function ($q) {
                        $q->where('user_id', Auth::guard('web')->id());
                    });
            })
            ->firstOrFail();

        if ((int)$business->owner_user_id !== (int)Auth::guard('web')->id() && (int)$userId !== (int)Auth::guard('web')->id()) {
             return back()->withErrors(['error' => 'Only the owner can remove other members.']);
        }

        if ((int)$business->owner_user_id === (int)$userId) {
             return back()->withErrors(['error' => 'Owners cannot leave the business. Transfer ownership first.']);
        }

        BusinessMembership::where('business_id', $business->id)
            ->where('user_id', $userId)
            ->delete();

        // Notifications
        try {
            $user = User::find($userId);
            if ($user) {
                // Determine if self-removal (leaving) or forced removal
                $isSelfRemoval = (int)$userId === (int)Auth::guard('web')->id();
                
                // Notify the user who was removed/left
                Mail::to($user->email)->send(new BusinessMemberRemovedEmail($business, $user, $isSelfRemoval));
                $this->fcmService->sendToUser(
                    $user->id,
                    $isSelfRemoval ? 'You left the business' : 'Access Removed',
                    $isSelfRemoval ? "You left {$business->name}." : "Your access to {$business->name} was removed.",
                    ['type' => 'business_access_removed', 'business_id' => $business->id]
                );

                // If user left voluntarily, notify the owner
                if ($isSelfRemoval) {
                    $owner = User::find($business->owner_user_id);
                    if ($owner) {
                        $this->fcmService->sendToUser(
                            $owner->id,
                            'Member Left',
                            "{$user->full_name} left your business {$business->name}.",
                            ['type' => 'business_member_left', 'business_id' => $business->id]
                        );
                    }
                }
            }
        } catch (\Throwable $e) {
            \Log::warning('Failed to send remove notifications', ['error' => $e->getMessage()]);
        }

        return back()->with('success', 'Member removed successfully.');
    }

    public function transferOwnership(Request $request): RedirectResponse
    {
        $businessId = session('selected_business_id');
        
        if (!$businessId) {
            return back()->withErrors(['error' => 'No business selected']);
        }

        $business = Business::where('id', $businessId)
            ->where(function ($query) {
                $query->where('owner_user_id', Auth::guard('web')->id())
                    ->orWhereHas('memberships', function ($q) {
                        $q->where('user_id', Auth::guard('web')->id());
                    });
            })
            ->firstOrFail();

        if ((int)$business->owner_user_id !== (int)Auth::guard('web')->id()) {
            abort(403, 'Unauthorized action. Only the owner can transfer ownership.');
        }

        $request->validate(['new_owner_email' => 'required|email']);
        
        $email = strtolower(trim($request->new_owner_email));
        $newOwner = User::where('email', $email)->first();

        if (!$newOwner) {
            return back()->withErrors(['new_owner_email' => 'User not found.']);
        }

        if ($newOwner->id === $business->owner_user_id) {
             return back()->withErrors(['new_owner_email' => 'User is already owner.']);
        }
        
        $oldOwnerId = $business->owner_user_id;

        // Create membership for new owner if not exists
        $membership = BusinessMembership::firstOrCreate([
            'business_id' => $business->id,
            'user_id' => $newOwner->id
        ], [
             'role' => BusinessMembership::ROLE_OWNER,
             'added_by_user_id' => Auth::guard('web')->id(),
             'joined_at' => now()
        ]);
        
        $membership->update(['role' => BusinessMembership::ROLE_OWNER]);

        // Downgrade old owner
        BusinessMembership::where('business_id', $business->id)
             ->where('user_id', $oldOwnerId)
             ->update(['role' => BusinessMembership::ROLE_MANAGER]);

        $business->update(['owner_user_id' => $newOwner->id]);

        // Notifications
        try {
            // Notify New Owner
            Mail::to($newOwner->email)->send(new BusinessOwnershipTransferredNewOwnerEmail($business, $newOwner, User::find($oldOwnerId), Auth::guard('web')->user()));
            $this->fcmService->sendToUser(
                $newOwner->id,
                'Business Ownership',
                "You are now the owner of {$business->name}.",
                ['type' => 'business_ownership_gained', 'business_id' => $business->id]
            );

            // Notify Old Owner (Current User)
            Mail::to(User::find($oldOwnerId)->email)->send(new BusinessOwnershipTransferredOldOwnerEmail($business, User::find($oldOwnerId), $newOwner));
             // Optional: Self-notification via FCM? Probably duplicate if they just did the action.
        } catch (\Throwable $e) {
            \Log::warning('Failed to send transfer notifications', ['error' => $e->getMessage()]);
        }

        return redirect()->route('web.profile')->with('success', 'Ownership transferred. You are now a manager.');
    }

    public function acceptInvitation($token): RedirectResponse
    {
        $invite = BusinessInvitation::where('token', $token)->first();

        if (!$invite) {
             return redirect()->route('web.home')->withErrors(['error' => 'Invalid invitation token.']);
        }

        if (!$invite->isActive()) {
             return redirect()->route('web.home')->withErrors(['error' => 'Invitation expired or already accepted.']);
        }
        
        // Ensure the logged-in user matches the invited email
        if (strtolower(Auth::guard('web')->user()->email) !== strtolower($invite->email)) {
            return redirect()->route('web.home')->withErrors(['error' => 'This invitation is for a different email address.']);
        }

        // Create Membership
        BusinessMembership::updateOrCreate(
            [
                'business_id' => $invite->business_id,
                'user_id' => Auth::guard('web')->id(),
            ],
            [
                'role' => $invite->role,
                'joined_at' => now(),
                'added_by_user_id' => $invite->invited_by_user_id,
            ]
        );

        // Update User Role to Seller if currently just 'user'
        $user = Auth::guard('web')->user();
        if ($user->role === 'user') {
            $user->update(['role' => 'seller']);
        }

        // Mark as accepted
        $invite->update(['accepted_at' => now()]);

        // Select the business for the session
        session(['selected_business_id' => $invite->business_id]);

        return redirect()->route('web.business.edit')->with('success', 'Invitation accepted! You are now managing ' . $invite->business->name);
    }

    public function revokeInvitation($id): RedirectResponse
    {
        $businessId = session('selected_business_id');
        
        if (!$businessId) {
            return back()->withErrors(['error' => 'No business selected']);
        }

        $business = Business::where('id', $businessId)
            ->where('owner_user_id', Auth::guard('web')->id())
            ->firstOrFail();

        $invite = BusinessInvitation::where('business_id', $business->id)
            ->where('id', $id)
            ->firstOrFail();

        $invite->update(['revoked_at' => now()]);

        // Notifications
        try {
            Mail::to($invite->email)->send(new BusinessInvitationRevokedEmail($business, $invite));
            
            // Try to find user if they exist to send push
            $user = User::where('email', $invite->email)->first();
            if ($user) {
                $this->fcmService->sendToUser(
                    $user->id,
                    'Invitation Revoked',
                    "Your invitation to manage {$business->name} was revoked.",
                    ['type' => 'invitation_revoked', 'business_id' => $business->id]
                );
            }
        } catch (\Throwable $e) {
             \Log::warning('Failed to send revoke notifications', ['error' => $e->getMessage()]);
        }

        return back()->with('success', 'Invitation revoked successfully.');
    }
}
