<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Models\Business;
use App\Models\Listing;
use App\Models\UserFavoriteSearch;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class FavoriteController extends Controller
{
    /**
     * Display the user's favorites.
     */
    public function index()
    {
        $user = Auth::guard('web')->user();
        
        $favoriteListings = $user->favoriteListings()->with(['business', 'media'])->latest()->get();
        $favoriteBusinesses = $user->favoriteBusinesses()->latest()->get();
        $favoriteSearches = $user->favoriteSearches()->latest()->get();

        return view('web.favorites', compact('favoriteListings', 'favoriteBusinesses', 'favoriteSearches'));
    }

    /**
     * Toggle a listing in favorites.
     */
    public function toggleListing(Request $request): JsonResponse
    {
        $listingId = $request->input('listing_id');
        $user = Auth::guard('web')->user();

        if ($user->favoriteListings()->where('listing_id', $listingId)->exists()) {
            $user->favoriteListings()->detach($listingId);
            $status = 'removed';
            $message = 'Removed from favorites';
        } else {
            $user->favoriteListings()->attach($listingId);
            $status = 'added';
            $message = 'Added to favorites';
        }

        return response()->json([
            'status' => 'success',
            'action' => $status,
            'message' => $message
        ]);
    }

    /**
     * Toggle a business in favorites.
     */
    public function toggleBusiness(Request $request): JsonResponse
    {
        $businessId = $request->input('business_id');
        $user = Auth::guard('web')->user();

        if ($user->favoriteBusinesses()->where('business_id', $businessId)->exists()) {
            $user->favoriteBusinesses()->detach($businessId);
            $status = 'removed';
            $message = 'Unfollowed shop';
        } else {
            $user->favoriteBusinesses()->attach($businessId);
            $status = 'added';
            $message = 'Following shop';
        }

        return response()->json([
            'status' => 'success',
            'action' => $status,
            'message' => $message
        ]);
    }

    /**
     * Save a search.
     */
    public function saveSearch(Request $request): JsonResponse
    {
        $user = Auth::guard('web')->user();
        
        $query = $request->input('q');
        $filters = $request->except(['_token', 'q', 'g-recaptcha-response']); // Exclude unnecessary fields
        
        // Basic validation: Must have at least a query or filter
        if (empty($query) && empty($filters)) {
             return response()->json([
                'status' => 'error',
                'message' => 'Please enter a search term or select filters to save.'
            ], 400);
        }

        // Check if already exists exactly
        $existing = $user->favoriteSearches()
            ->where('query', $query)
            ->where('filters', json_encode($filters))
            ->first();

        if ($existing) {
             return response()->json([
                'status' => 'info',
                'message' => 'Search already saved'
            ]);
        }

        $user->favoriteSearches()->create([
            'name' => $query ?: 'Custom Search', // Default name, user can allow renaming later if needed
            'query' => $query,
            'filters' => $filters,
        ]);

        return response()->json([
            'status' => 'success',
            'message' => 'Search saved successfully'
        ]);
    }

    /**
     * Remove a saved search.
     */
    public function removeSearch($id): JsonResponse
    {
        $user = Auth::guard('web')->user();
        
        $deleted = $user->favoriteSearches()->where('id', $id)->delete();

        if ($deleted) {
            return response()->json([
                'status' => 'success',
                'message' => 'Saved search removed'
            ]);
        }

        return response()->json([
            'status' => 'error',
            'message' => 'Failed to remove search'
        ], 400);
    }
}
