<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use Illuminate\Support\Facades\Mail;
use App\Mail\SupportQuery;
use App\Models\BlogPost;
use App\Models\NewsletterSubscription;

class HelpController extends Controller
{
    /**
     * Display the About Us page.
     */
    public function about()
    {
        return view('web.about');
    }

    /**
     * Display the Help Center landing page.
     */
    public function index()
    {
        return view('web.help-center');
    }

    /**
     * Display the "How to Buy" guide.
     */
    public function howToBuy()
    {
        return view('web.guides.how-to-buy');
    }

    /**
     * Display the "How to Sell" guide.
     */
    public function howToSell()
    {
        return view('web.guides.how-to-sell');
    }

    /**
     * Display the Safety Center page.
     */
    public function safetyCenter()
    {
        return view('web.safety-center');
    }

    /**
     * Display the FAQs page.
     */
    public function faqs()
    {
        $faqs = [
            ['q' => 'Where can I download the Yilick app?', 'a' => 'Yilick is available on all platforms! You can download it for Android on the Google Play Store, for iOS on the Apple App Store, or download the APK directly from our website. We also have a fast web app accessible from any browser.'],
            ['q' => 'Is Yilick free to use?', 'a' => 'Yes! Listing your items on Yilick is completely free for standard ads. We also offer premium "Hot Deals" and "Promo" options if you want your ad to sell faster by reaching more buyers.'],
            ['q' => 'How do I buy an item safely?', 'a' => 'Find an item, click on it to see details, and use the "Chat" or "Call" buttons. We recommend meeting in person in safe public places like malls, gas stations, or busy cafes during daylight hours.'],
            ['q' => 'How do I pay the seller?', 'a' => 'Most sellers in Uganda prefer Cash on Delivery or Mobile Money (MTN MoMo/Airtel Money). We strongly advise NEVER paying before seeing and inspecting the item properly.'],
            ['q' => 'Is delivery included?', 'a' => 'Delivery is usually arranged between the buyer and the seller. Many sellers offer delivery for an extra fee, while others prefer physical meetups. Always discuss this clearly before finalizing the deal.'],
            ['q' => 'How do I sell my used items?', 'a' => 'Click the "Sell" button (plus icon), select the right category, upload clear photos (take them in good light!), write a descriptive title (e.g., "Used iPhone 13 128GB - Great Condition"), set your price, and publish!'],
            ['q' => 'What if I see a fake item or scam?', 'a' => 'Click the "Report" button on the listing page immediately. You can also use our Contact Support form to report fraudulent behavior. We take your safety very seriously and will investigate all reports.'],
            ['q' => 'How long does my ad stay active?', 'a' => 'Standard ads are active for 30 days. You can renew them from your "My Ads" section if they haven\'t sold yet.'],
            ['q' => 'Can I have multiple shops?', 'a' => 'Yes, Yilick allows you to create and manage multiple business profiles under one account, which is great if you sell different types of products or have multiple store branches.'],
            ['q' => 'Are there any hidden fees or commissions?', 'a' => 'No, Yilick does not charge commissions on sales. The price you negotiate with the seller is what you pay. We only charge for optional ad promotions.'],
            ['q' => 'How do I verify my phone number?', 'a' => 'Go to your Profile -> Settings -> Account. You can request a verification code sent via SMS to confirm your number and build trust with buyers.'],
            ['q' => 'What items are strictly prohibited?', 'a' => 'We prohibit illegal substances, firearms, stolen goods, pornographic material, and endangered species. Please read our full Terms of Service for a detailed list.'],
            ['q' => 'Can I sell services like plumbing or construction?', 'a' => 'Absolutely! We have a dedicated "Services" category. Make sure to describe your expertise clearly and provide a portfolio of your work if possible.'],
            ['q' => 'How do I edit or delete my listing?', 'a' => 'Go to "My Ads" on your profile. Each ad has an "Edit" and "Delete" button. Changes are reflected instantly on the marketplace.'],
            ['q' => 'What is a "Hot Deal"?', 'a' => 'A Hot Deal is a premium promotion that places your ad in a special highlighted section at the top of the homepage and search results, giving you maximum visibility.'],
            ['q' => 'How do I report a stolen item?', 'a' => 'If you find an item you believe is stolen, report the ad immediately using the "Report" button and contact local authorities with the listing details.'],
            ['q' => 'Can I use Yilick without an account?', 'a' => 'You can browse and search for items without an account, but you will need to log in to chat with sellers, save favorites, or post your own ads.'],
            ['q' => 'How do I change my location on a listing?', 'a' => 'When creating or editing an ad, you can select the specific district and neighborhood where the item is located to help local buyers find you.'],
            ['q' => 'What should I do if a seller doesn\'t answer my chat?', 'a' => 'Wait for a reasonable time, as some sellers may be busy. If they provide a phone number, you can try calling them directly for a faster response.'],
            ['q' => 'How do I upload better photos?', 'a' => 'Use natural daylight, keep the background clean, and show the item from multiple angles. High-quality photos significantly increase your chances of a quick sale.'],
            ['q' => 'Is Yilick available in all Ugandan districts?', 'a' => 'Yes! From Kampala to Gulu, Mbarara to Mbale, Yilick is available nationwide. Simply set your location to see items near you.'],
            ['q' => 'What is the "Verification Badge"?', 'a' => 'A verification badge is awarded to users who have confirmed their identity and have a track record of successful transactions. It helps build credibility.'],
            ['q' => 'How do I reset my password?', 'a' => 'Click "Forgot Password" on the Sign In page. We will send you a secure link to your registered email to set a new password.'],
        ];

        return view('web.faqs', compact('faqs'));
    }

    /**
     * Handle the support email submission.
     */
    public function sendSupportEmail(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'subject' => 'required|string|max:255',
            'message' => 'required|string|max:2000',
        ]);

        try {
            Mail::to(config('support.email'))->send(new SupportQuery($validated));
            return back()->with('success', 'Your message has been sent successfully. We will get back to you soon!');
        } catch (\Exception $e) {
            return back()->with('error', 'Sorry, there was an error sending your message. Please try again later.');
        }
    }

    /**
     * Display the Terms of Service page.
     */
    public function terms()
    {
        return view('web.terms');
    }

    /**
     * Display the Privacy Policy page.
     */
    public function privacy()
    {
        return view('web.privacy');
    }

    /**
     * Display the Cookie Policy page.
     */
    public function cookie()
    {
        return view('web.cookie');
    }

    /**
     * Display the Contact Us page.
     */
    public function contact()
    {
        return view('web.contact');
    }

    /**
     * Display the Careers page.
     */
    public function careers()
    {
        return view('web.careers');
    }

    /**
     * Display the Blog page.
     */
    public function blog(Request $request)
    {
        $search = $request->input('search');
        $category = $request->input('category');

        $query = BlogPost::where('is_published', true);

        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('excerpt', 'like', "%{$search}%")
                  ->orWhere('content', 'like', "%{$search}%")
                  ->orWhere('category', 'like', "%{$search}%");
            });
        }

        if ($category) {
            $query->where('category', $category);
        }

        $featuredPost = (clone $query)->where('is_featured', true)
            ->latest('published_at')
            ->first();

        $posts = $query->where('id', '!=', $featuredPost?->id ?? 0)
            ->latest('published_at')
            ->paginate(9)
            ->withQueryString();

        return view('web.blog', compact('featuredPost', 'posts'));
    }

    /**
     * Display a specific blog post.
     */
    public function showBlogPost(BlogPost $blogPost)
    {
        if (!$blogPost->is_published) {
            abort(404);
        }

        $relatedPosts = BlogPost::where('is_published', true)
            ->where('category', $blogPost->category)
            ->where('id', '!=', $blogPost->id)
            ->latest('published_at')
            ->limit(3)
            ->get();

        return view('web.blog-details', compact('blogPost', 'relatedPosts'));
    }

    /**
     * Handle the newsletter subscription submission.
     */
    public function subscribeNewsletter(Request $request)
    {
        $validated = $request->validate([
            'email' => 'required|email|max:255',
        ]);

        $subscriber = NewsletterSubscription::firstOrCreate(
            ['email' => $validated['email']],
            ['is_active' => true]
        );

        if ($request->ajax()) {
            return response()->json([
                'status' => 'success',
                'message' => 'Thank you for joining the Yilick Journal! You are now part of Uganda\'s most forward-thinking digital commerce community.'
            ]);
        }

        return back()->with('success', 'Thanks for subscribing! You will receive our latest updates soon.');
    }
}
