<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use App\Models\UserSubscription;
use App\Models\PremiumPackage;

class MembershipController extends Controller
{
    public function index(Request $request)
    {
        // Auto-login from Mobile App
        if ($request->has('auth_token')) {
            try {
                $user = auth('api')->setToken($request->auth_token)->user();
                if ($user) {
                    auth('web')->login($user);
                }
            } catch (\Exception $e) { }
        }
            
        $user = auth()->user();
        $activeSubscription = null;
        $businessId = $request->input('business_id');

        if ($user) {
            if ($businessId) {
                $business = \App\Models\Business::find($businessId);
                if ($business) {
                    $activeSubscription = $business->activeSubscription();
                }
            } else {
                $activeSubscription = UserSubscription::where('user_id', $user->id)
                    ->where('status', 'active')
                    ->where('starts_at', '<=', now())
                    ->where('expires_at', '>', now())
                    ->latest('expires_at')
                    ->first();
            }
        }
        
        $packagesQuery = PremiumPackage::where('is_active', true)
            ->orderBy('sort_order', 'asc')
            ->orderBy('price', 'asc');

        // If user has a PAID subscription (not free virtual one), hide the free package
        if ($activeSubscription && $activeSubscription->package && $activeSubscription->package->slug !== 'free') {
            $packagesQuery->where('slug', '!=', 'free');
        }

        $packages = $packagesQuery->get();
            
        if ($request->input('mode') === 'mobile') {
            return view('mobile.membership.index', compact('packages', 'activeSubscription', 'businessId'));
        }

        return view('web.membership.index', compact('packages', 'activeSubscription', 'businessId'));
    }

    public function checkout(Request $request, $id)
    {
        if (!auth('web')->check()) {
            return redirect()->route('login');
        }

        $package = PremiumPackage::where('is_active', true)->findOrFail($id);
        $businessId = $request->input('business_id');
        $user = auth('web')->user();

        // Calculate discount for upgrades
        $discount = 0;
        $activeSubscription = null;
        $isUpgrade = false;

        if ($businessId) {
            $activeSubscription = UserSubscription::where('business_id', $businessId)
                ->where('status', 'active')
                ->where('expires_at', '>', now())
                ->first();

            if ($activeSubscription) {
                $currentPackage = $activeSubscription->package;
                
                // If new package is more expensive, it's an upgrade
                if ($package->price > $currentPackage->price) {
                    $isUpgrade = true;
                    $remainingDays = now()->diffInDays($activeSubscription->expires_at, false);
                    if ($remainingDays > 0) {
                        $dailyRate = $currentPackage->price / ($currentPackage->duration_days ?: 30);
                        $discount = $remainingDays * $dailyRate;
                        
                        // Cap discount to package price
                        $discount = min($discount, (float) $package->price);
                    }
                }
            }
        }

        $finalPrice = max(0, $package->price - $discount);
            
        if ($request->input('mode') === 'mobile') {
            return view('mobile.membership.checkout', compact('package', 'businessId', 'activeSubscription', 'discount', 'finalPrice', 'isUpgrade'));
        }

        return view('web.membership.checkout', compact('package', 'businessId', 'activeSubscription', 'discount', 'finalPrice', 'isUpgrade'));
    }

    public function processPayment(Request $request, $id)
    {
        if (!auth('web')->check()) {
            return redirect()->route('login');
        }

        $package = PremiumPackage::findOrFail($id);
        $user = auth('web')->user();

        $request->validate([
            'full_phone' => ['required', 'string', 'regex:/^\+256[0-9]{9}$/'],
        ], [
            'full_phone.regex' => 'Please provide a valid Uganda mobile money number (+256...)',
        ]);

        $phoneNumber = $request->input('full_phone');
        $reference = Str::random(20);
        $businessId = $request->input('business_id');

        // Calculate final price again (security check)
        $discount = 0;
        if ($businessId) {
            $activeSubscription = UserSubscription::where('business_id', $businessId)
                ->where('status', 'active')
                ->where('expires_at', '>', now())
                ->first();

            if ($activeSubscription && $package->price > $activeSubscription->package->price) {
                $currentPackage = $activeSubscription->package;
                $remainingDays = now()->diffInDays($activeSubscription->expires_at, false);
                if ($remainingDays > 0) {
                    $dailyRate = $currentPackage->price / ($currentPackage->duration_days ?: 30);
                    $discount = $remainingDays * $dailyRate;
                }
            }
        }

        $finalPrice = max(0, (float) $package->price - $discount);

        // Create a pending subscription
        $subscription = UserSubscription::create([
            'user_id' => $user->id,
            'business_id' => $businessId,
            'package_id' => $package->id,
            'status' => 'pending',
            'amount_paid' => $finalPrice,
            'payment_method' => 'mobile_money',
            'transaction_id' => $reference,
            'notes' => $discount > 0 ? "Upgrade discount applied: " . number_format($discount) : null,
        ]);

        try {
            $response = Http::withHeaders([
                'Accept' => 'application/vnd.relworx.v2',
                'Authorization' => 'Bearer ' . config('services.relworx.api_key'),
            ])->post(config('services.relworx.base_url') . 'mobile-money/request-payment', [
                'account_no' => config('services.relworx.account_no'),
                'reference' => $reference,
                'msisdn' => $phoneNumber,
                'currency' => 'UGX',
                'amount' => (float) $finalPrice,
                'description' => "Subscription for {$package->name} - {$user->name}",
            ]);

            if ($response->successful() && $response->json('success')) {
                $subscription->update([
                    'notes' => 'Internal Reference: ' . $response->json('internal_reference'),
                ]);
                
                // Redirect to specialized processing screen
                $redirectRoute = route('web.membership.processing', ['transaction_id' => $reference]);
                if ($request->input('mode') === 'mobile') {
                    $redirectRoute .= '?mode=mobile';
                }
                
                return redirect($redirectRoute);
            }

            $subscription->update(['status' => 'failed']);
            Log::error('Relworx Payment Error: ' . $response->body());
            return back()->withInput()->withErrors(['full_phone' => 'Payment failed: ' . ($response->json('message') ?? 'Unknown error')]);

        } catch (\Exception $e) {
            $subscription->update(['status' => 'failed']);
            Log::error('Relworx API Exception: ' . $e->getMessage());
            return back()->withInput()->withErrors(['full_phone' => 'An error occurred while processing your payment. Please try again.']);
        }
    }

    public function processing(Request $request, $transaction_id)
    {
        $subscription = UserSubscription::where('transaction_id', $transaction_id)->firstOrFail();
        
        if ($request->input('mode') === 'mobile') {
            return view('mobile.membership.processing', compact('subscription'));
        }
        return view('web.membership.processing', compact('subscription'));
    }

    public function success(Request $request)
    {
        if ($request->input('mode') === 'mobile') {
            return view('mobile.membership.success');
        }
        return view('web.membership.success');
    }

    public function failed(Request $request)
    {
        if ($request->input('mode') === 'mobile') {
            return view('mobile.membership.failed');
        }
        return view('web.membership.failed');
    }

    public function checkStatus($transaction_id)
    {
        $subscription = UserSubscription::where('transaction_id', $transaction_id)->firstOrFail();
        
        return response()->json([
            'status' => $subscription->status,
            'is_active' => $subscription->status === 'active'
        ]);
    }
}
