<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class NotificationController extends Controller
{
    /**
     * Display a listing of the notifications.
     */
    public function index(Request $request)
    {
        $user = Auth::guard('web')->user();
        
        $query = \Illuminate\Notifications\DatabaseNotification::query()
            ->where(function ($q) use ($user) {
                $q->where('notifiable_id', $user->id)
                  ->where('notifiable_type', get_class($user));
            })
            ->orWhereNull('notifiable_id')
            ->latest();

        $allNotifications = $query->paginate(20);
        
        // Count unread (this is tricky for global without pivot, so we might just count user-specific unread for badges)
        // For the view lists, we'll keep the standard counts for now or update logic if needed.
        // But for distinct lists:
        $unreadNotifications = (clone $query)->whereNull('read_at')->get();
        $readNotifications = (clone $query)->whereNotNull('read_at')->get();

        return view('web.notifications', [
            'allNotifications' => $allNotifications,
            'unreadNotifications' => $unreadNotifications,
            'readNotifications' => $readNotifications,
        ]);
    }

    /**
     * Mark a specific notification as read.
     */
    public function markAsRead($id)
    {
        $notification = Auth::guard('web')->user()->notifications()->findOrFail($id);
        $notification->markAsRead();

        if (request()->wantsJson()) {
            return response()->json(['success' => true]);
        }

        return back()->with('success', 'Notification marked as read.');
    }

    /**
     * Mark all notifications as read.
     */
    public function markAllAsRead()
    {
        Auth::guard('web')->user()->unreadNotifications->markAsRead();

        if (request()->wantsJson()) {
            return response()->json(['success' => true]);
        }

        return back()->with('success', 'All notifications marked as read.');
    }
}
