<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ProfileController extends Controller
{
    /**
     * Display the user's profile page.
     */
    public function index()
    {
        $user = Auth::guard('web')->user();
        
        $businesses = $user->allBusinesses()->get();
        
        // Get or initialize selected business
        $businessId = session('selected_business_id');
        if (!$businessId) {
            $firstBusiness = $businesses->first();
            if ($firstBusiness) {
                $businessId = $firstBusiness->id;
                session(['selected_business_id' => $businessId]);
            }
        }
        
        // Filter stats by selected business
        $listingsCount = $user->createdListings()
            ->when($businessId, fn($q) => $q->where('business_id', $businessId))
            ->where('status', \App\Models\Listing::STATUS_PUBLISHED)
            ->count();
            
        $favoritesCount = $user->favoriteListings()->count();
        
        return view('web.profile', [
            'user' => $user,
            'businesses' => $businesses,
            'listingsCount' => $listingsCount,
            'favoritesCount' => $favoritesCount,
        ]);
    }

    /**
     * Display the user's ads management page.
     */
    public function myAds(Request $request)
    {
        $user = Auth::guard('web')->user();

        // Redirect to business creation if the user has no shop
        if ($user->allBusinesses()->count() === 0) {
            return redirect()->route('business.create')
                ->with('info', 'Please create your shop profile to manage your ads.');
        }

        $tab = $request->input('tab', 'published');
        
        // Get or initialize selected business
        $businessId = session('selected_business_id');
        if (!$businessId) {
            $firstBusiness = $user->allBusinesses()->first();
            if ($firstBusiness) {
                $businessId = $firstBusiness->id;
                session(['selected_business_id' => $businessId]);
            }
        }

        // If no business exists, they shouldn't see ads for any (or we can redirect)
        // For now, filter by the ID if it exists, otherwise an empty result set is fine
        
        // Query listings for the selected business (all listings for that business, not just user's)
        $baseQuery = \App\Models\Listing::where('business_id', $businessId);
        
        // Counts
        $publishedCount = (clone $baseQuery)
            ->where('status', \App\Models\Listing::STATUS_PUBLISHED)
            ->count();
            
        $reviewCount = (clone $baseQuery)
            ->where('status', \App\Models\Listing::STATUS_REVIEW)
            ->count();
            
        $declinedCount = (clone $baseQuery)
            ->where('status', \App\Models\Listing::STATUS_REJECTED)
            ->count();
            
        $closedCount = (clone $baseQuery)
            ->whereIn('status', ['expired', 'sold', 'closed', 'draft'])
            ->count();

        // Active Listings Query
        $query = (clone $baseQuery)->with(['media', 'business']);

        switch($tab) {
            case 'review':
                $query->where('status', \App\Models\Listing::STATUS_REVIEW);
                break;
            case 'declined':
                $query->where('status', \App\Models\Listing::STATUS_REJECTED);
                break;
            case 'closed':
                $query->whereIn('status', ['expired', 'sold', 'closed', 'draft']);
                break;
            case 'published':
            default:
                $query->where('status', \App\Models\Listing::STATUS_PUBLISHED);
                $tab = 'published';
                break;
        }

        $listings = $query->latest()->paginate(12)->withQueryString();

        return view('web.my-ads', compact('listings', 'publishedCount', 'reviewCount', 'declinedCount', 'closedCount', 'tab'));
    }

    public function switchBusiness(Request $request)
    {
        $request->validate([
            'business_id' => 'required|exists:businesses,id',
        ]);

        $business = Auth::user()->allBusinesses()->where('id', $request->business_id)->firstOrFail();

        session(['selected_business_id' => $business->id]);

        if ($request->wantsJson()) {
            // Get updated stats for this business
            $listingsCount = Auth::user()->createdListings()
                ->where('business_id', $business->id)
                ->where('status', \App\Models\Listing::STATUS_PUBLISHED)
                ->count();
                
            return response()->json([
                'success' => true,
                'message' => "Switched to {$business->name}",
                'listingsCount' => $listingsCount,
                'businessName' => $business->name
            ]);
        }

        return back()->with('success', "Switched to {$business->name}");
    }
}
