<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Business;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class SettingsController extends Controller
{
    public function index(): View
    {
        $user = Auth::guard('web')->user();
        
        // Get selected business logic similar to ProfileController
        $businessId = session('selected_business_id');
        $business = null;

        if ($businessId) {
            $business = $user->allBusinesses()->find($businessId);
        }

        // Fallback or default business logic if needed, but for settings 
        // we might just show settings for the *selected* business or user account.
        // If no business is selected, we might want to select the first one.
        if (!$business) {
             $business = $user->allBusinesses()->first();
             if ($business) {
                 session(['selected_business_id' => $business->id]);
             }
        }

        return view('web.settings', compact('user', 'business'));
    }

    public function update(Request $request)
    {
        $user = Auth::guard('web')->user();
        $businessId = session('selected_business_id');
        
        if (!$businessId) {
             return back()->with('error', 'No business selected.');
        }

        $business = $user->allBusinesses()->findOrFail($businessId);

        $validated = $request->validate([
            'is_feedback_enabled' => 'nullable|boolean',
            'is_chat_enabled' => 'nullable|boolean',
        ]);

        // Explicitly handle checkbox/boolean updates
        // If a toggle is sent via AJAX or form, we update it.
        // For simplicity with this specific request which usually comes via AJAX/toggle
        
        if ($request->has('is_feedback_enabled')) {
            $business->is_feedback_enabled = $request->boolean('is_feedback_enabled');
        }

        if ($request->has('is_chat_enabled')) {
            $business->is_chat_enabled = $request->boolean('is_chat_enabled');
        }

        $business->save();

        if ($request->wantsJson()) {
             return response()->json(['success' => true, 'message' => 'Settings updated successfully.']);
        }

        return back()->with('success', 'Settings updated successfully.');
    }

    public function personalDetails(): View
    {
        return view('web.settings.personal-details');
    }

    public function updatePersonalDetails(Request $request): RedirectResponse
    {
        $user = Auth::guard('web')->user();

        $validated = $request->validate([
            'full_name' => 'required|string|min:2|regex:/[a-zA-Z]/',
            'date_of_birth' => 'nullable|date|before:13 years ago',
            'gender' => 'nullable|string|in:male,female,other,prefer_not_to_say',
            'profile_photo' => 'nullable|image|max:2048', // 2MB max
        ], [
            'full_name.regex' => 'Name must contain at least one letter.',
            'date_of_birth.before' => 'You must be at least 13 years old.',
        ]);

        $user->fill([
            'full_name' => $validated['full_name'],
            'date_of_birth' => $validated['date_of_birth'] ?? null,
            'gender' => $validated['gender'] ?? null,
        ]);

        if ($request->hasFile('profile_photo')) {
            $path = $request->file('profile_photo')->store('profile-photos', 'public');
            $user->profile_photo = $path;
        }

        $user->save();

        return back()->with('success', 'Personal details updated successfully.');
    }

    public function changePassword(): View
    {
        return view('web.settings.change-password');
    }

    public function updatePassword(Request $request): RedirectResponse
    {
        $user = Auth::guard('web')->user();

        $validated = $request->validate([
            'current_password' => 'required',
            'password' => 'required|string|min:6|confirmed',
        ], [
            'current_password.required' => 'Current password is required.',
            'password.required' => 'New password is required.',
            'password.min' => 'Password must be at least 6 characters.',
            'password.confirmed' => 'Passwords do not match.',
        ]);

        // Verify current password
        if (!\Hash::check($validated['current_password'], $user->password)) {
            return back()->withErrors(['current_password' => 'Current password is incorrect.']);
        }

        // Update password
        $user->password = $validated['password'];
        $user->save();

        return back()->with('success', 'Password changed successfully.');
    }
}
