<?php

namespace App\Http\Middleware;

use App\Models\ActivityLog;
use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class ActivityLogger
{
    public function handle(Request $request, Closure $next): Response
    {
        $start = microtime(true);

        $response = null;
        $exception = null;

        try {
            $response = $next($request);
            return $response;
        } catch (\Throwable $e) {
            $exception = $e;
            throw $e;
        } finally {
            $durationMs = (int) round((microtime(true) - $start) * 1000);

            try {
                $route = $request->route();
                $routeName = $route?->getName();
                $controllerAction = $route?->getActionName();

                $statusCode = null;
                if ($response instanceof Response) {
                    $statusCode = $response->getStatusCode();
                }

                if ($exception && $statusCode === null) {
                    $statusCode = 500;
                }

                $user = null;
                try {
                    $user = auth('api')->user() ?: auth('web')->user();
                } catch (\Throwable $e) {
                    $user = null;
                }

                $payload = $this->sanitizePayload($request);

                $method = strtoupper($request->getMethod());
                $path = '/' . ltrim($request->path(), '/');

                $activity = $method . ' ' . $path;
                $action = $routeName ?: $controllerAction;

                ActivityLog::create([
                    'user_id' => $user?->id,
                    'action' => $action,
                    'activity' => $activity,
                    'method' => $method,
                    'path' => $path,
                    'route_name' => $routeName,
                    'controller_action' => $controllerAction,
                    'status_code' => $statusCode,
                    'duration_ms' => $durationMs,
                    'ip_address' => $request->ip(),
                    'user_agent' => $request->userAgent(),
                    'payload' => $payload,
                ]);

                // Update user's last active status
                if ($user && (!$user->last_login_at || $user->last_login_at->diffInMinutes(now()) >= 5)) {
                    $user->update(['last_login_at' => now()]);
                }
            } catch (\Throwable $e) {
                // Avoid breaking the request lifecycle if logging fails.
                \Log::warning('ActivityLogger failed', [
                    'error' => $e->getMessage(),
                ]);
            }
        }
    }

    protected function sanitizePayload(Request $request): array
    {
        $sensitiveKeys = [
            'password',
            'password_confirmation',
            'token',
            'refresh_token',
            'otp',
            'code',
            'authorization',
        ];

        $input = $request->except($sensitiveKeys);

        foreach ($sensitiveKeys as $key) {
            if ($request->has($key)) {
                $input[$key] = '[REDACTED]';
            }
        }

        return [
            'query' => $request->query(),
            'input' => $input,
        ];
    }
}
