<?php

namespace App\Http\Requests\User;

use App\Models\User;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class UpdateProfileRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        $userId = $this->user()->id;
        $phoneRegex = '/^\+?[1-9]\d{1,14}$/';

        return [
            // Profile
            'full_name' => ['sometimes', 'string', 'max:255'],
            'username' => [
                'sometimes',
                'nullable',
                'string',
                'max:50',
                'regex:/^[a-zA-Z0-9_]+$/',
                Rule::unique('users', 'username')->ignore($userId),
            ],
            'bio' => ['sometimes', 'nullable', 'string', 'max:1000'],
            'date_of_birth' => ['sometimes', 'nullable', 'date', 'before:today', 'after:1900-01-01'],
            'gender' => ['sometimes', 'nullable', 'string', 'in:' . implode(',', [
                User::GENDER_MALE,
                User::GENDER_FEMALE,
                User::GENDER_OTHER,
                User::GENDER_PREFER_NOT_TO_SAY,
            ])],
            'profile_photo' => [
                'sometimes',
                'nullable',
                'image',
                'mimes:jpeg,jpg,png,gif,webp',
                'max:5120',
                'dimensions:min_width=100,min_height=100,max_width=4096,max_height=4096',
            ],

            // Contact
            'phone' => [
                'sometimes',
                'nullable',
                'string',
                'max:20',
                "regex:{$phoneRegex}",
                Rule::unique('users', 'phone')->ignore($userId),
            ],
            'secondary_phone' => ['sometimes', 'nullable', 'string', 'max:20', "regex:{$phoneRegex}"],
            'whatsapp_number' => ['sometimes', 'nullable', 'string', 'max:20', "regex:{$phoneRegex}"],
            'secondary_whatsapp_number' => ['sometimes', 'nullable', 'string', 'max:20', "regex:{$phoneRegex}"],

            // Location
            'country' => ['sometimes', 'nullable', 'string', 'max:100'],
            'country_code' => ['sometimes', 'nullable', 'string', 'size:2', 'alpha'],
            'state' => ['sometimes', 'nullable', 'string', 'max:100'],
            'city' => ['sometimes', 'nullable', 'string', 'max:100'],
            'district' => ['sometimes', 'nullable', 'string', 'max:100'],
            'address' => ['sometimes', 'nullable', 'string', 'max:255'],
            'postal_code' => ['sometimes', 'nullable', 'string', 'max:20'],
            'latitude' => ['sometimes', 'nullable', 'numeric', 'between:-90,90'],
            'longitude' => ['sometimes', 'nullable', 'numeric', 'between:-180,180'],
            'timezone' => ['sometimes', 'nullable', 'string', 'max:50', 'timezone:all'],

            // Social Links
            'website' => ['sometimes', 'nullable', 'url', 'max:255'],
            'facebook_url' => ['sometimes', 'nullable', 'url', 'max:255'],
            'twitter_url' => ['sometimes', 'nullable', 'url', 'max:255'],
            'instagram_url' => ['sometimes', 'nullable', 'url', 'max:255'],
            'linkedin_url' => ['sometimes', 'nullable', 'url', 'max:255'],
            'tiktok_url' => ['sometimes', 'nullable', 'url', 'max:255'],
            'youtube_url' => ['sometimes', 'nullable', 'url', 'max:255'],

            // Preferences
            'language' => ['sometimes', 'string', 'max:10'],
            'currency' => ['sometimes', 'string', 'size:3', 'alpha'],
            'is_notifications_enabled' => ['sometimes', 'boolean'],
            'is_email_notifications_enabled' => ['sometimes', 'boolean'],
            'is_sms_notifications_enabled' => ['sometimes', 'boolean'],
            'is_push_notifications_enabled' => ['sometimes', 'boolean'],
            'is_marketing_emails_enabled' => ['sometimes', 'boolean'],
            'show_phone_publicly' => ['sometimes', 'boolean'],
            'show_email_publicly' => ['sometimes', 'boolean'],
            'show_location_publicly' => ['sometimes', 'boolean'],
        ];
    }

    public function messages(): array
    {
        return [
            'username.regex' => 'Username can only contain letters, numbers, and underscores.',
            'phone.regex' => 'The phone number must be in E.164 format (e.g., +1234567890).',
            'secondary_phone.regex' => 'The secondary phone number must be in E.164 format.',
            'whatsapp_number.regex' => 'The WhatsApp number must be in E.164 format.',
            'profile_photo.max' => 'The profile photo must not exceed 5MB.',
            'profile_photo.dimensions' => 'The profile photo must be between 100x100 and 4096x4096 pixels.',
            'latitude.between' => 'Latitude must be between -90 and 90.',
            'longitude.between' => 'Longitude must be between -180 and 180.',
            'country_code.size' => 'Country code must be exactly 2 characters (e.g., US, GB).',
            'currency.size' => 'Currency code must be exactly 3 characters (e.g., USD, EUR).',
        ];
    }
}
