<?php

namespace App\Jobs;

use App\Models\Listing;
use App\Notifications\AdPublishedNotification;
use App\Notifications\AdRejectedNotification;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class AutoReviewListing implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(public int $listingId)
    {
    }

    public function handle(): void
    {
        \Log::info('AutoReviewListing starting', ['listing_id' => $this->listingId]);
        
        $listing = Listing::with('createdBy')->find($this->listingId);

        if (!$listing || $listing->status !== Listing::STATUS_REVIEW) {
            return;
        }

        $title = strtolower($listing->title);
        $description = strtolower($listing->description);
        $fullContent = $title . ' ' . $description;

        // 1. Check for Prohibited / Illegal Items
        $prohibitedTerms = [
            'drug', 'marijuana', 'weed', 'cocaine', 'firearm', 'gun', 'weapon', 
            'explosive', 'kidney', 'organ', 'blood', 'escort', 
            'porn', 'adult content', 'gambling', 'betting tips'
        ];
        if ($this->containsAny($fullContent, $prohibitedTerms)) {
            $this->reject($listing, 'prohibited_item');
            return;
        }

        // 2. Check for Fraud / Scam / Sensitive Info Requests
        $fraudTerms = [
            'otp', 'pin code', 'password', 'login details', 'bank account details',
            'share your card', 'send money first', 'advance payment', 'processing fee',
            'winning prize', 'lottery', 'inheritance'
        ];
        if ($this->containsAny($fullContent, $fraudTerms)) {
            $this->reject($listing, 'fraud_scam'); // or sensitive_info_request depending on context
            return;
        }

        // 3. Check for Contact Details / Links in Forbidden Places (Title)
        // Regex for phone numbers (basic)
        $phoneRegex = '/(\+?[0-9]{1,3})?[-. ]?([0-9]{3})[-. ]?([0-9]{3})[-. ]?([0-9]{4,})/';
        $emailRegex = '/[a-z0-9._%+-]+@[a-z0-9.-]+\.[a-z]{2,}/i';
        // Broad URL regex to catch www. and common TLDs even without http
        $urlRegex = '/(https?:\/\/|www\.|[a-z0-9.-]+\.(com|net|org|biz|info|ug|co\.ug|online|shop))(?!\.yilick\.ug)/i';

        if (preg_match($phoneRegex, $listing->title) || preg_match($emailRegex, $listing->title) || preg_match($urlRegex, $listing->title)) {
            $this->reject($listing, 'contact_in_forbidden_format');
            return;
        }

        // 4. Check for External Links in Description
        if (preg_match($urlRegex, $listing->description)) {
            $this->reject($listing, 'disallowed_links');
            return;
        }

        // Catch words like "website", "click here", "visit my"
        $redirectTerms = ['visit my website', 'click here', 'www.', 'http', '.com', '.ug'];
        if ($this->containsAny($fullContent, $redirectTerms)) {
             $this->reject($listing, 'disallowed_links');
             return;
        }

        // 5. Basic Quality Checks
        // Vague Description
        if (strlen($listing->description) < 15) {
            $this->reject($listing, 'vague_description');
            return;
        }

        // Unclear / Spammy Title
        if (strlen($listing->title) < 5 || preg_match('/[!@#$%^&*()]{4,}/', $listing->title)) {
            $this->reject($listing, 'unclear_title');
            return;
        }

        // Missing Price
        if (empty($listing->price_amount) || $listing->price_amount <= 0) {
            $this->reject($listing, 'price_missing');
            return;
        }

        // 6. Duplicate Listing Check (Basic: same user, same title in last 24h)
        $isDuplicate = Listing::where('created_by_user_id', $listing->created_by_user_id)
            ->where('id', '!=', $listing->id)
            ->where('title', $listing->title)
            ->where('created_at', '>=', now()->subDay())
            ->exists();

        if ($isDuplicate) {
            $this->reject($listing, 'duplicate_listing');
            return;
        }

        // 7. Check Subscription Limit before Publishing
        $subscription = $listing->business->subscription;
        // Re-fetch to get fresh count
        if ($subscription) {
            $subscription->refresh(); 
            // We use 'can_create_listing' or manual check. 
            // Note: getListingsUsedAttribute now includes REVIEW status which this listing currently has.
            // So if limit is 5 and we have 5 used (including this one), it's fine.
            // But if we have 6 used, we should block.
            
            $maxListings = $subscription->package->max_listings ?? 0;
            $usedListings = $subscription->listings_used; // This counts 'published' and 'review'

            // If we are strictly enforcing, and used > max, we should revert this one to draft.
            // However, this listing is ALREADY in 'review' status, so it is already counted in $usedListings.
            // So if $usedListings > $maxListings, it means we are over limit.
            
            if ($usedListings > $maxListings) {
                $listing->update([
                    'status' => Listing::STATUS_DRAFT,
                    'decline_reason' => 'Subscription limit reached.'
                ]);
                
                $listing->createdBy->notify(new \App\Notifications\AdLimitReachedNotification($listing));
                
                \Log::info('AutoReviewListing completed (limit reached -> draft)', ['listing_id' => $this->listingId]);
                return;
            }
        }

        // 8. Auto-Approve if everything looks fine
        $listing->update([
            'status' => Listing::STATUS_PUBLISHED,
            'published_at' => now(),
        ]);

        $listing->createdBy->notify(new AdPublishedNotification($listing));
        
        \Log::info('AutoReviewListing completed (published)', ['listing_id' => $this->listingId]);
    }

    private function containsAny(string $content, array $terms): bool
    {
        foreach ($terms as $term) {
            if (str_contains($content, $term)) {
                return true;
            }
        }
        return false;
    }

    private function reject(Listing $listing, string $reason): void
    {
        $listing->update([
            'status' => Listing::STATUS_REJECTED,
            'decline_reason' => $reason
        ]);
        
        $listing->createdBy->notify(new AdRejectedNotification($listing));
        
        \Log::info('AutoReviewListing completed (rejected)', ['listing_id' => $this->listingId, 'reason' => $reason]);
    }
}
