<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Hash;

class OtpVerification extends Model
{
    public const TYPE_EMAIL = 'email';
    public const TYPE_PHONE = 'phone';
    public const TYPE_PHONE_PRIMARY = 'phone_primary';
    public const TYPE_PHONE_SECONDARY = 'phone_secondary';
    public const TYPE_WHATSAPP = 'whatsapp';
    public const TYPE_WHATSAPP_PRIMARY = 'whatsapp_primary';
    public const TYPE_WHATSAPP_SECONDARY = 'whatsapp_secondary';

    public const MAX_ATTEMPTS = 5;
    public const EXPIRY_MINUTES = 10;

    protected $fillable = [
        'identifier',
        'type',
        'otp_hash',
        'attempts',
        'expires_at',
        'verified_at',
    ];

    protected function casts(): array
    {
        return [
            'expires_at' => 'datetime',
            'verified_at' => 'datetime',
        ];
    }

    /**
     * Generate a new OTP for the given identifier.
     */
    public static function generateFor(string $identifier, string $type): array
    {
        $otp = str_pad((string) random_int(0, 999999), 6, '0', STR_PAD_LEFT);

        static::where('identifier', $identifier)
            ->where('type', $type)
            ->whereNull('verified_at')
            ->delete();

        static::create([
            'identifier' => $identifier,
            'type' => $type,
            'otp_hash' => Hash::make($otp),
            'expires_at' => now()->addMinutes(self::EXPIRY_MINUTES),
        ]);

        return [
            'otp' => $otp,
            'expires_in' => self::EXPIRY_MINUTES * 60,
        ];
    }

    /**
     * Verify the OTP for the given identifier.
     */
    public static function verify(string $identifier, string $type, string $otp): bool
    {
        $verification = static::where('identifier', $identifier)
            ->where('type', $type)
            ->whereNull('verified_at')
            ->where('expires_at', '>', now())
            ->where('attempts', '<', self::MAX_ATTEMPTS)
            ->latest()
            ->first();

        if (!$verification) {
            return false;
        }

        $verification->increment('attempts');

        if (!Hash::check($otp, $verification->otp_hash)) {
            return false;
        }

        $verification->update(['verified_at' => now()]);

        return true;
    }

    /**
     * Check if the identifier has a pending OTP.
     */
    public static function hasPending(string $identifier, string $type): bool
    {
        return static::where('identifier', $identifier)
            ->where('type', $type)
            ->whereNull('verified_at')
            ->where('expires_at', '>', now())
            ->exists();
    }

    /**
     * Clean up expired OTPs.
     */
    public static function cleanExpired(): int
    {
        return static::where('expires_at', '<=', now())->delete();
    }
}
