<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class PosOrder extends Model
{
    protected $fillable = [
        'business_id',
        'cashier_user_id',
        'total_amount',
        'tax_amount',
        'discount_amount',
        'payment_method',
        'items',
        'synced_at',
        'offline_uuid',
        'pos_customer_id',
        'receipt_number',
        'status',
        'refunded_amount',
        'ai_summary',
    ];

    protected $casts = [
        'items' => 'array',
        'synced_at' => 'datetime',
        'total_amount' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'refunded_amount' => 'decimal:2',
    ];

    public function business(): BelongsTo
    {
        return $this->belongsTo(Business::class);
    }

    public function refunds(): \Illuminate\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(PosRefund::class);
    }

    public function customer(): BelongsTo
    {
        return $this->belongsTo(PosCustomer::class, 'pos_customer_id');
    }

    public function payments(): \Illuminate\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(PosPayment::class);
    }

    public function cashier(): BelongsTo
    {
        return $this->belongsTo(User::class, 'cashier_user_id');
    }

    public function ledger(): \Illuminate\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(PosCustomerLedger::class, 'pos_order_id');
    }

    protected $appends = ['primary_payment_method'];

    public function getPrimaryPaymentMethodAttribute(): string
    {
        // If the 'payment_method' column is already set to something specific (legacy or direct), use it.
        // Otherwise, inspect the payments relation.
        if (!empty($this->attributes['payment_method']) && strtolower($this->attributes['payment_method']) !== 'split') {
             return $this->attributes['payment_method'];
        }

        if ($this->relationLoaded('payments')) {
             if ($this->payments->count() === 1) {
                 return $this->payments->first()->method ?? 'cash';
             }
             if ($this->payments->count() > 1) {
                 return 'mixed';
             }
        }
        
        return $this->attributes['payment_method'] ?? 'cash';
    }
}
