<?php

namespace App\Models;

use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Concerns\HasUuids;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use PHPOpenSourceSaver\JWTAuth\Contracts\JWTSubject;

class User extends Authenticatable implements JWTSubject, MustVerifyEmail
{
    use HasFactory, Notifiable, SoftDeletes;

    // Status Constants
    public const STATUS_ACTIVE = 'active';
    public const STATUS_SUSPENDED = 'suspended';
    public const STATUS_BANNED = 'banned';
    public const STATUS_PENDING = 'pending';
    public const STATUS_DELETED = 'deleted';

    // Role Constants
    public const ROLE_USER = 'user';
    public const ROLE_SELLER = 'seller';
    public const ROLE_PREMIUM_SELLER = 'premium_seller';
    public const ROLE_MODERATOR = 'moderator';
    public const ROLE_ADMIN = 'admin';
    public const ROLE_SUPER_ADMIN = 'super_admin';

    // Gender Constants
    public const GENDER_MALE = 'male';
    public const GENDER_FEMALE = 'female';
    public const GENDER_OTHER = 'other';
    public const GENDER_PREFER_NOT_TO_SAY = 'prefer_not_to_say';

    // Identity Verification Status Constants
    public const IDENTITY_UNVERIFIED = 'unverified';
    public const IDENTITY_PENDING = 'pending';
    public const IDENTITY_VERIFIED = 'verified';
    public const IDENTITY_REJECTED = 'rejected';

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        // Profile
        'uuid',
        'profile_photo',
        'full_name',
        'username',
        'bio',
        'date_of_birth',
        'gender',
        
        // Contact
        'email',
        'phone',
        'secondary_phone',
        'whatsapp_number',
        'secondary_whatsapp_number',
        'is_email_verified',
        'is_phone_verified',
        'email_verified_at',
        'phone_verified_at',
        'secondary_phone_verified_at',
        'whatsapp_verified_at',
        'secondary_whatsapp_verified_at',
        
        // Location
        'country',
        'country_code',
        'state',
        'city',
        'district',
        'address',
        'postal_code',
        'latitude',
        'longitude',
        'timezone',
        
        // Social
        'website',
        'facebook_url',
        'twitter_url',
        'instagram_url',
        'linkedin_url',
        'tiktok_url',
        'youtube_url',
        
        // Security
        'password',
        'two_factor_enabled',
        'two_factor_secret',
        'two_factor_recovery_codes',
        'password_changed_at',
        'failed_login_attempts',
        'locked_until',
        'last_login_at',
        'last_login_ip',
        
        // Preferences
        'language',
        'currency',
        'is_notifications_enabled',
        'is_email_notifications_enabled',
        'is_sms_notifications_enabled',
        'is_push_notifications_enabled',
        'is_marketing_emails_enabled',
        'show_phone_publicly',
        'show_email_publicly',
        'show_location_publicly',
        
        // Identity Verification
        'identity_verification_status',
        'identity_document_type',
        'identity_document_number',
        'identity_document_photo',
        
        // Status & Role
        'status',
        'suspension_reason',
        'suspended_at',
        'suspension_ends_at',
        'role',
        
        // Referral
        'referral_code',
        'referred_by',
        'referral_count',
        
        // Metadata
        'registration_source',
        'registration_ip',
        'device_type',
        'metadata',
        'fcm_token',

        // Legacy Support
        'v1_salt',
        'v1_password_hash',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
        'two_factor_secret',
        'two_factor_recovery_codes',
        'identity_document_number',
        'identity_document_photo',
    ];

    /**
     * The model's default values for attributes.
     *
     * @var array
     */
    protected $attributes = [
        'is_email_verified' => false,
        'is_phone_verified' => false,
        'two_factor_enabled' => false,
        'failed_login_attempts' => 0,
        'is_notifications_enabled' => true,
        'is_email_notifications_enabled' => true,
        'is_sms_notifications_enabled' => true,
        'is_push_notifications_enabled' => true,
        'is_marketing_emails_enabled' => false,
        'show_phone_publicly' => false,
        'show_email_publicly' => false,
        'show_location_publicly' => true,
        'identity_verification_status' => 'unverified',
        'status' => 'active',
        'role' => 'user',
        'referral_count' => 0,
        'language' => 'en',
        'currency' => 'USD',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'phone_verified_at' => 'datetime',
            'secondary_phone_verified_at' => 'datetime',
            'whatsapp_verified_at' => 'datetime',
            'secondary_whatsapp_verified_at' => 'datetime',
            'identity_verified_at' => 'datetime',
            'password_changed_at' => 'datetime',
            'locked_until' => 'datetime',
            'last_login_at' => 'datetime',
            'suspended_at' => 'datetime',
            'suspension_ends_at' => 'datetime',
            'date_of_birth' => 'date',
            'password' => 'hashed',
            'is_email_verified' => 'boolean',
            'is_phone_verified' => 'boolean',
            'two_factor_enabled' => 'boolean',
            'is_notifications_enabled' => 'boolean',
            'is_email_notifications_enabled' => 'boolean',
            'is_sms_notifications_enabled' => 'boolean',
            'is_push_notifications_enabled' => 'boolean',
            'is_marketing_emails_enabled' => 'boolean',
            'show_phone_publicly' => 'boolean',
            'show_email_publicly' => 'boolean',
            'show_location_publicly' => 'boolean',
            'latitude' => 'decimal:8',
            'longitude' => 'decimal:8',
            'metadata' => 'array',
        ];
    }

    /**
     * Bootstrap the model and its traits.
     */
    protected static function boot(): void
    {
        parent::boot();

        static::creating(function (User $user) {
            if (empty($user->uuid)) {
                $user->uuid = (string) Str::uuid();
            }
            if (empty($user->referral_code)) {
                $user->referral_code = self::generateReferralCode();
            }
        });
    }

    /**
     * Generate a unique referral code.
     */
    public static function generateReferralCode(): string
    {
        do {
            $code = strtoupper(Str::random(8));
        } while (self::withTrashed()->where('referral_code', $code)->exists());

        return $code;
    }

    /**
     * Get the route key for the model.
     */
    public function getRouteKeyName(): string
    {
        return 'uuid';
    }

    /**
     * Get the identifier that will be stored in the subject claim of the JWT.
     */
    public function getJWTIdentifier(): mixed
    {
        return $this->getKey();
    }

    /**
     * Return a key value array, containing any custom claims to be added to the JWT.
     */
    public function getJWTCustomClaims(): array
    {
        return [
            'uuid' => $this->uuid,
            'role' => $this->role,
            'status' => $this->status,
        ];
    }

    /**
     * Get the profile photo URL.
     */
    public function getProfilePhotoUrlAttribute(): ?string
    {
        if (!$this->profile_photo) {
            return asset('assets/icons/favicon.png');
        }
        return Storage::disk('public')->url($this->profile_photo);
    }

    /**
     * Get the full location string.
     */
    public function getFullLocationAttribute(): ?string
    {
        $parts = array_filter([
            $this->address,
            $this->district,
            $this->city,
            $this->state,
            $this->country,
        ]);

        return !empty($parts) ? implode(', ', $parts) : null;
    }

    /**
     * Get coordinates as array.
     */
    public function getCoordinatesAttribute(): ?array
    {
        if ($this->latitude && $this->longitude) {
            return [
                'lat' => (float) $this->latitude,
                'lng' => (float) $this->longitude,
            ];
        }
        return null;
    }

    /**
     * Check if user is active.
     */
    public function isActive(): bool
    {
        return $this->status === self::STATUS_ACTIVE;
    }

    /**
     * Check if user is suspended.
     */
    public function isSuspended(): bool
    {
        return $this->status === self::STATUS_SUSPENDED;
    }

    /**
     * Check if user is banned.
     */
    public function isBanned(): bool
    {
        return $this->status === self::STATUS_BANNED;
    }

    /**
     * Check if user account is locked.
     */
    public function isLocked(): bool
    {
        return $this->locked_until && $this->locked_until->isFuture();
    }

    /**
     * Check if user is admin.
     */
    public function isAdmin(): bool
    {
        return in_array($this->role, [self::ROLE_ADMIN, self::ROLE_SUPER_ADMIN]);
    }

    /**
     * Check if user is super admin.
     */
    public function isSuperAdmin(): bool
    {
        return $this->role === self::ROLE_SUPER_ADMIN;
    }

    /**
     * Check if user is moderator.
     */
    public function isModerator(): bool
    {
        return $this->role === self::ROLE_MODERATOR;
    }

    /**
     * Check if user is seller.
     */
    public function isSeller(): bool
    {
        return in_array($this->role, [self::ROLE_SELLER, self::ROLE_PREMIUM_SELLER]);
    }

    /**
     * Check if user is premium seller.
     */
    public function isPremiumSeller(): bool
    {
        return $this->role === self::ROLE_PREMIUM_SELLER;
    }

    /**
     * Check if user has verified identity.
     */
    public function hasVerifiedIdentity(): bool
    {
        return $this->identity_verification_status === self::IDENTITY_VERIFIED;
    }

    /**
     * Check if user has verified email.
     */
    public function hasVerifiedEmail(): bool
    {
        return $this->is_email_verified || !is_null($this->email_verified_at);
    }

    /**
     * Check if user has verified phone.
     */
    public function hasVerifiedPhone(): bool
    {
        return $this->is_phone_verified || !is_null($this->phone_verified_at);
    }

    /**
     * Mark the user's email as verified.
     */
    public function markEmailAsVerified(): bool
    {
        return $this->forceFill([
            'email_verified_at' => $this->freshTimestamp(),
            'is_email_verified' => true,
        ])->save();
    }

    /**
     * Mark the user's phone as verified.
     */
    public function markPhoneAsVerified(): bool
    {
        return $this->forceFill([
            'phone_verified_at' => $this->freshTimestamp(),
            'is_phone_verified' => true,
        ])->save();
    }

    /**
     * Record a failed login attempt.
     */
    public function recordFailedLogin(): void
    {
        $this->increment('failed_login_attempts');

        if ($this->failed_login_attempts >= 5) {
            $this->update([
                'locked_until' => now()->addMinutes(30),
            ]);
        }
    }

    /**
     * Record a successful login.
     */
    public function recordSuccessfulLogin(string $ip): void
    {
        $this->update([
            'failed_login_attempts' => 0,
            'locked_until' => null,
            'last_login_at' => now(),
            'last_login_ip' => $ip,
        ]);
    }

    /**
     * Suspend the user.
     */
    public function suspend(string $reason, ?\DateTimeInterface $until = null): bool
    {
        return $this->update([
            'status' => self::STATUS_SUSPENDED,
            'suspension_reason' => $reason,
            'suspended_at' => now(),
            'suspension_ends_at' => $until,
        ]);
    }

    /**
     * Unsuspend the user.
     */
    public function unsuspend(): bool
    {
        return $this->update([
            'status' => self::STATUS_ACTIVE,
            'suspension_reason' => null,
            'suspended_at' => null,
            'suspension_ends_at' => null,
        ]);
    }

    /**
     * Get the user who referred this user.
     */
    public function referrer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'referred_by');
    }

    /**
     * Get users referred by this user.
     */
    public function referrals(): HasMany
    {
        return $this->hasMany(User::class, 'referred_by');
    }

    public function businessFeedbacks(): HasMany
    {
        return $this->hasMany(BusinessFeedback::class);
    }

    public function businessFeedbackResponses(): HasMany
    {
        return $this->hasMany(BusinessFeedbackResponse::class, 'responded_by_user_id');
    }

    public function subscriptions(): HasMany
    {
        return $this->hasMany(UserSubscription::class);
    }

    public function createdListings(): HasMany
    {
        return $this->hasMany(Listing::class, 'created_by_user_id');
    }

    public function ownedBusinesses(): HasMany
    {
        return $this->hasMany(Business::class, 'owner_user_id');
    }

    /**
     * Get businesses where the user is a member (not owner).
     */
    public function memberBusinesses(): BelongsToMany
    {
        return $this->belongsToMany(Business::class, 'business_memberships', 'user_id', 'business_id')
            ->withPivot(['role', 'joined_at'])
            ->withTimestamps();
    }

    /**
     * Get all businesses (owned + member) for the user.
     */
    public function allBusinesses()
    {
        $ownedIds = $this->ownedBusinesses()->pluck('id');
        $memberIds = $this->memberBusinesses()->pluck('businesses.id');
        
        return Business::whereIn('id', $ownedIds->merge($memberIds)->unique());
    }

    public function followedBusinesses(): BelongsToMany
    {
        return $this->belongsToMany(Business::class, 'business_follows')
            ->withTimestamps();
    }

    public function favoriteListings(): BelongsToMany
    {
        return $this->belongsToMany(Listing::class, 'user_favorite_listings')
            ->withTimestamps();
    }

    public function favoriteBusinesses(): BelongsToMany
    {
        return $this->belongsToMany(Business::class, 'user_favorite_businesses')
            ->withTimestamps();
    }

    public function favoriteSearches(): HasMany
    {
        return $this->hasMany(UserFavoriteSearch::class);
    }

    public function conversations(): HasMany
    {
        return $this->hasMany(Conversation::class, 'buyer_id');
    }

    public function messages(): HasMany
    {
        return $this->hasMany(Message::class, 'sender_id');
    }

    /**
     * Get the FCM tokens for the user's devices.
     */
    public function fcmTokens(): HasMany
    {
        return $this->hasMany(UserFcmToken::class);
    }

    /**
     * Get the refresh tokens for the user.
     */
    public function refreshTokens(): HasMany
    {
        return $this->hasMany(JwtRefreshToken::class);
    }

    /**
     * Scope a query to only include active users.
     */
    public function scopeActive($query)
    {
        return $query->where('status', self::STATUS_ACTIVE);
    }

    /**
     * Scope a query to only include users with a specific role.
     */
    public function scopeWithRole($query, string $role)
    {
        return $query->where('role', $role);
    }

    /**
     * Scope a query to filter by location.
     */
    public function scopeInCountry($query, string $countryCode)
    {
        return $query->where('country_code', $countryCode);
    }

    /**
     * Scope a query to filter by city.
     */
    public function scopeInCity($query, string $city)
    {
        return $query->where('city', $city);
    }

    /**
     * Scope a query to find users near coordinates.
     */
    public function scopeNearby($query, float $latitude, float $longitude, float $radiusKm = 50)
    {
        $haversine = "(6371 * acos(cos(radians(?)) * cos(radians(latitude)) * cos(radians(longitude) - radians(?)) + sin(radians(?)) * sin(radians(latitude))))";

        return $query->whereNotNull('latitude')
            ->whereNotNull('longitude')
            ->whereRaw("{$haversine} < ?", [$latitude, $longitude, $latitude, $radiusKm])
            ->orderByRaw("{$haversine}", [$latitude, $longitude, $latitude]);
    }

    /**
     * Scope a query to filter by identity verification status.
     */
    public function scopeIdentityVerified($query)
    {
        return $query->where('identity_verification_status', self::IDENTITY_VERIFIED);
    }

    /**
     * Send the email verification notification.
     * Override to use custom welcome email instead of default.
     */
    public function sendEmailVerificationNotification()
    {
        // Don't send anything here - we handle it via SendWelcomeEmail job in AuthController
        // This prevents the default Laravel verification email from being sent
    }

    public function reportedListings(): HasMany
    {
        return $this->hasMany(ListingReport::class);
    }

    public function blockedBusinesses(): HasMany
    {
        return $this->hasMany(BusinessBlock::class);
    }
}
