<?php

namespace App\Notifications;

use App\Models\UserSubscription;
use App\Services\FcmService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class SubscriptionActivatedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(public UserSubscription $subscription)
    {
    }

    public function via($notifiable): array
    {
        return ['mail', 'database'];
    }

    public function toMail($notifiable): MailMessage
    {
        $package = $this->subscription->package;
        $business = $this->subscription->business;
        
        return (new MailMessage)
            ->subject('Subscription Activated: ' . ($package->name ?? 'Premium Plan'))
            ->greeting('Hello ' . ($notifiable->full_name ?? $notifiable->username) . '!')
            ->line('Your subscription to the **' . ($package->name ?? 'Premium Plan') . '** has been successfully activated for **' . ($business->name ?? 'your business') . '**.')
            ->line('You now have access to all premium features, including increased listing limits and enhanced search visibility.')
            ->line('Expiry Date: ' . $this->subscription->expires_at->format('M d, Y'))
            ->action('Go to Dashboard', config('app.url') . '/membership')
            ->line('Thank you for being a part of ' . config('app.name') . '!')
            ->salutation('Best regards,  
The ' . config('app.name') . ' Team');
    }

    public function toArray($notifiable): array
    {
        $package = $this->subscription->package;
        $title = 'Subscription Activated!';
        $body = 'Your ' . ($package->name ?? 'Premium') . ' plan is now active for ' . ($this->subscription->business->name ?? 'your business') . '.';
        
        $data = [
            'type' => 'subscription_activated',
            'subscription_id' => $this->subscription->id,
            'package_name' => $package->name ?? 'Premium',
        ];

        // Trigger FCM
        if ($notifiable->id) {
            try {
                app(FcmService::class)->sendToUser(
                    $notifiable->id,
                    $title,
                    $body,
                    $data
                );
            } catch (\Throwable $e) {
                \Log::warning('Failed to send FCM notification (SubscriptionActivated)', [
                    'user_id' => $notifiable->id,
                    'error' => $e->getMessage(),
                ]);
            }
        }

        return [
            'title' => $title,
            'body' => $body,
            'data' => $data,
        ];
    }
}
