<?php

namespace App\Services;

use App\Models\Ad;
use App\Models\AdTracking;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AdTrackerService
{
    /**
     * Track an ad event
     * 
     * @param Ad $ad
     * @param string $type ('view', 'click', 'impression')
     * @param Request $request
     */
    public function trackEvent(Ad $ad, string $type, Request $request): void
    {
        $userId = auth('api')->id();
        $ipAddress = $request->ip();
        $userAgent = $request->userAgent();
        $sessionId = $this->getSessionId($request);

        // Deduplication logic for views and impressions (24h)
        if (in_array($type, ['view', 'impression'])) {
            $isDuplicate = $this->isDuplicateTracking($ad->id, $type, $userId, $sessionId, $ipAddress);
            if ($isDuplicate) {
                return;
            }
        }

        DB::transaction(function () use ($ad, $type, $userId, $ipAddress, $userAgent, $sessionId) {
            AdTracking::create([
                'ad_id' => $ad->id,
                'user_id' => $userId,
                'ip_address' => $ipAddress,
                'user_agent' => $userAgent,
                'session_id' => $sessionId,
                'type' => $type,
                'tracked_at' => now(),
            ]);

            $column = "{$type}s_count";
            $ad->increment($column);
        });
    }

    protected function getSessionId(Request $request): ?string
    {
        try {
            return $request->session()->getId();
        } catch (\RuntimeException $e) {
            return md5($request->ip() . $request->userAgent());
        }
    }

    protected function isDuplicateTracking(int $adId, string $type, ?int $userId, ?string $sessionId, ?string $ipAddress): bool
    {
        $query = AdTracking::where('ad_id', $adId)
            ->where('type', $type)
            ->where('tracked_at', '>', now()->subHours(24));

        if ($userId) {
            $query->where('user_id', $userId);
        } else {
            $query->where(function ($q) use ($sessionId, $ipAddress) {
                if ($sessionId) {
                    $q->where('session_id', $sessionId);
                }
                if ($ipAddress) {
                    $q->orWhere('ip_address', $ipAddress);
                }
            });
        }

        return $query->exists();
    }
}
