<?php

namespace App\Services;

use App\Models\Business;
use App\Models\User;
use Google\Client;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class FcmService
{
    protected $projectId;
    protected $serviceAccountPath;

    public function __construct()
    {
        $this->serviceAccountPath = config('services.fcm.service_account');
        $this->projectId = config('services.fcm.project_id');

        // Auto-detect project ID from service account file if not set or to ensure consistency
        if (file_exists($this->serviceAccountPath)) {
            $json = json_decode(file_get_contents($this->serviceAccountPath), true);
            if (isset($json['project_id'])) {
                $this->projectId = $json['project_id'];
            }
        }
    }

    /**
     * Send a notification to all registered devices of a user.
     */
    public function sendToUser(int $userId, string $title, string $body, array $data = [])
    {
        $user = User::with('fcmTokens')->find($userId);
        if (!$user || $user->fcmTokens->isEmpty()) {
            return false;
        }

        $tokens = $user->fcmTokens->pluck('fcm_token')->toArray();
        return $this->sendToMultipleTokens($tokens, $title, $body, $data);
    }

    /**
     * Send a notification to the owner and all members of a business.
     */
    public function sendToBusiness(int $businessId, string $title, string $body, array $data = [])
    {
        $business = Business::with(['memberships.user.fcmTokens', 'owner.fcmTokens'])->find($businessId);
        if (!$business) {
            return false;
        }

        $tokenList = [];

        // Owner tokens
        if ($business->owner && $business->owner->fcmTokens->isNotEmpty()) {
            foreach ($business->owner->fcmTokens as $token) {
                $tokenList[] = $token->fcm_token;
            }
        }

        // Member tokens
        foreach ($business->memberships as $membership) {
            if ($membership->user && $membership->user->fcmTokens->isNotEmpty()) {
                foreach ($membership->user->fcmTokens as $token) {
                    $tokenList[] = $token->fcm_token;
                }
            }
        }

        $uniqueTokens = array_unique($tokenList);
        if (empty($uniqueTokens)) {
            return false;
        }

        return $this->sendToMultipleTokens($uniqueTokens, $title, $body, $data);
    }

    /**
     * Send a push notification to a specific list of FCM tokens.
     */
    public function sendToMultipleTokens(array $tokens, string $title, string $body, array $data = [])
    {
        $success = true;
        foreach ($tokens as $token) {
            if (!$this->sendNotification($token, $title, $body, $data)) {
                $success = false;
            }
        }
        return $success;
    }

    /**
     * Send a push notification to a specific FCM token.
     */
    public function sendNotification(string $token, string $title, string $body, array $data = [])
    {
        if (empty($token)) {
            return false;
        }

        try {
            $accessToken = $this->getAccessToken();
            
            if (!$accessToken) {
                Log::error('FCM: Failed to get access token. Check if service-account.json exists and is valid.');
                return false;
            }

            $url = "https://fcm.googleapis.com/v1/projects/{$this->projectId}/messages:send";

            $payload = [
                'message' => [
                    'token' => $token,
                    'notification' => [
                        'title' => $title,
                        'body' => $body,
                    ],
                    'data' => array_map('strval', $data),
                    'android' => [
                        'priority' => 'high',
                        'notification' => [
                            'sound' => 'default',
                        ],
                    ],
                    'apns' => [
                        'payload' => [
                            'aps' => [
                                'sound' => 'default',
                                'badge' => 1,
                            ],
                        ],
                    ],
                ],
            ];

            $response = Http::withToken($accessToken)
                ->post($url, $payload);

            if ($response->successful()) {
                return true;
            }

            Log::error('FCM: Failed to send notification', [
                'status' => $response->status(),
                'response' => $response->json(),
                'token' => substr($token, 0, 10) . '...'
            ]);

            return false;
        } catch (\Exception $e) {
            Log::error('FCM: Exception while sending notification', [
                'message' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Get the OAuth2 access token for FCM V1.
     */
    protected function getAccessToken()
    {
        if (!file_exists($this->serviceAccountPath)) {
            return null;
        }

        try {
            $client = new Client();
            $client->setAuthConfig($this->serviceAccountPath);
            $client->addScope('https://www.googleapis.com/auth/firebase.messaging');
            
            $token = $client->fetchAccessTokenWithAssertion();
            
            return $token['access_token'] ?? null;
        } catch (\Exception $e) {
            Log::error('FCM: Error fetching access token', ['message' => $e->getMessage()]);
            return null;
        }
    }
}
