<?php

namespace App\Services;

use Google\Client;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class FirebaseService
{
    protected $projectId;
    protected $serviceAccountPath;
    protected $databaseUrl = 'https://yilick-inc-default-rtdb.firebaseio.com';

    public function __construct()
    {
        $this->serviceAccountPath = config('services.fcm.service_account');
        
        if (file_exists($this->serviceAccountPath)) {
            $json = json_decode(file_get_contents($this->serviceAccountPath), true);
            if (isset($json['project_id'])) {
                $this->projectId = $json['project_id'];
            }
        }
    }

    /**
     * Get company info from Firebase RTDB.
     */
    public function getCompanyInfo(): array
    {
        try {
            $accessToken = $this->getAccessToken();
            if (!$accessToken) {
                return [];
            }

            $response = Http::withToken($accessToken)
                ->get("{$this->databaseUrl}/app_config.json");

            if ($response->successful()) {
                return $response->json() ?? [];
            }

            Log::error('Firebase RTDB: Failed to fetch data', ['status' => $response->status()]);
            return [];
        } catch (\Exception $e) {
            Log::error('Firebase RTDB: Exception fetching data', ['message' => $e->getMessage()]);
            return [];
        }
    }

    /**
     * Update company info in Firebase RTDB.
     */
    public function updateCompanyInfo(array $data): bool
    {
        try {
            $accessToken = $this->getAccessToken();
            if (!$accessToken) {
                return false;
            }

            // We use PATCH to update specific fields without wiping others if needed, 
            // but the requirement implies updating the whole structure. 
            // PUT replaces the data at the path.
            $response = Http::withToken($accessToken)
                ->patch("{$this->databaseUrl}/app_config.json", $data);

            if ($response->successful()) {
                return true;
            }

            Log::error('Firebase RTDB: Failed to update data', [
                'status' => $response->status(), 
                'body' => $response->body()
            ]);
            return false;
        } catch (\Exception $e) {
            Log::error('Firebase RTDB: Exception updating data', ['message' => $e->getMessage()]);
            return false;
        }
    }

    /**
     * Get the OAuth2 access token for Firebase Database.
     */
    protected function getAccessToken()
    {
        if (!file_exists($this->serviceAccountPath)) {
            return null;
        }

        try {
            $client = new Client();
            $client->setAuthConfig($this->serviceAccountPath);
            $client->addScope('https://www.googleapis.com/auth/firebase.database');
            $client->addScope('https://www.googleapis.com/auth/userinfo.email');
            
            $token = $client->fetchAccessTokenWithAssertion();
            
            return $token['access_token'] ?? null;
        } catch (\Exception $e) {
            Log::error('Firebase RTDB: Error fetching access token', ['message' => $e->getMessage()]);
            return null;
        }
    }
}
