<?php

namespace App\Services;

use Illuminate\Support\Facades\Storage;

class ImageProcessingService
{
    /**
     * Process an uploaded image: compress, create watermarked version, and detect tags
     * 
     * @param string $originalPath Path to original uploaded file
     * @param string $businessName Business name for watermark
     * @return array ['compressed_path' => string, 'watermarked_path' => string, 'tags' => array]
     */
    public function processImage(string $originalPath, string $businessName): array
    {
        $fullPath = Storage::disk('public')->path($originalPath);
        
        // Load image
        $imageInfo = getimagesize($fullPath);
        if (!$imageInfo) {
            return ['compressed_path' => $originalPath, 'watermarked_path' => null];
        }

        $mimeType = $imageInfo['mime'];
        $image = $this->loadImage($fullPath, $mimeType);
        
        if (!$image) {
            return ['compressed_path' => $originalPath, 'watermarked_path' => null];
        }

        // Get original dimensions
        $originalWidth = imagesx($image);
        $originalHeight = imagesy($image);

        // Compress image (resize if too large, quality 85)
        $maxWidth = 1920;
        $maxHeight = 1920;
        
        if ($originalWidth > $maxWidth || $originalHeight > $maxHeight) {
            $ratio = min($maxWidth / $originalWidth, $maxHeight / $originalHeight);
            $newWidth = (int) ($originalWidth * $ratio);
            $newHeight = (int) ($originalHeight * $ratio);
            
            $compressedImage = imagecreatetruecolor($newWidth, $newHeight);
            
            // Preserve transparency for PNG
            if ($mimeType === 'image/png') {
                imagealphablending($compressedImage, false);
                imagesavealpha($compressedImage, true);
            }
            
            imagecopyresampled($compressedImage, $image, 0, 0, 0, 0, $newWidth, $newHeight, $originalWidth, $originalHeight);
            imagedestroy($image);
            $image = $compressedImage;
        }

        // Save compressed version (overwrite original)
        $this->saveImage($image, $fullPath, $mimeType, 85);
        
        // Create watermarked version
        $watermarkedImage = $this->cloneImage($image);
        $this->addWatermark($watermarkedImage, $businessName);
        
        // Generate watermarked path
        $pathInfo = pathinfo($originalPath);
        $watermarkedPath = $pathInfo['dirname'] . '/' . $pathInfo['filename'] . '_watermarked.' . $pathInfo['extension'];
        $watermarkedFullPath = Storage::disk('public')->path($watermarkedPath);
        
        // Save watermarked version
        $this->saveImage($watermarkedImage, $watermarkedFullPath, $mimeType, 85);
        
        imagedestroy($image);
        imagedestroy($watermarkedImage);

        // Detect image tags
        $recognitionService = new ImageRecognitionService();
        $tags = $recognitionService->detectLabels($originalPath);
        
        // Fallback to basic analysis if API fails or not configured
        if (empty($tags)) {
            $tags = $recognitionService->analyzeImageBasic($originalPath);
        }

        return [
            'compressed_path' => $originalPath,
            'watermarked_path' => $watermarkedPath,
            'tags' => $tags,
        ];
    }

    private function loadImage(string $path, string $mimeType)
    {
        return match ($mimeType) {
            'image/jpeg', 'image/jpg' => imagecreatefromjpeg($path),
            'image/png' => imagecreatefrompng($path),
            'image/gif' => imagecreatefromgif($path),
            'image/webp' => imagecreatefromwebp($path),
            default => null,
        };
    }

    private function saveImage($image, string $path, string $mimeType, int $quality = 85): void
    {
        match ($mimeType) {
            'image/jpeg', 'image/jpg' => imagejpeg($image, $path, $quality),
            'image/png' => imagepng($image, $path, (int) (9 - ($quality / 10))),
            'image/gif' => imagegif($image, $path),
            'image/webp' => imagewebp($image, $path, $quality),
            default => null,
        };
    }

    private function cloneImage($sourceImage)
    {
        $width = imagesx($sourceImage);
        $height = imagesy($sourceImage);
        
        $clone = imagecreatetruecolor($width, $height);
        
        // Preserve transparency
        imagealphablending($clone, false);
        imagesavealpha($clone, true);
        
        imagecopy($clone, $sourceImage, 0, 0, 0, 0, $width, $height);
        
        return $clone;
    }

    private function addWatermark($image, string $businessName): void
    {
        $width = imagesx($image);
        $height = imagesy($image);
        
        $appName = config('app.name', 'Yilick');
        
        // Watermark text
        $line1 = "Posted on {$appName}";
        $line2 = $businessName;
        
        // Dynamic font size: smaller for small images, larger for big images
        // Scale between 10px (for 500px images) and 24px (for 2000px+ images)
        $minSize = min($width, $height);
        $fontSize = max(10, min(24, (int) ($minSize * 0.02)));
        $fontPath = $this->getSystemFont();
        
        // Semi-transparent white text (alpha 40 = ~70% transparent)
        $textColor = imagecolorallocatealpha($image, 255, 255, 255, 40);
        
        if (empty($fontPath)) {
            // Fallback to built-in GD font if no TTF found
            $font = 5; // Largest built-in font
            $fontWidth = imagefontwidth($font);
            $fontHeight = imagefontheight($font);
            
            $line1Width = strlen($line1) * $fontWidth;
            $line2Width = strlen($line2) * $fontWidth;
            
            $line1X = (int) (($width - $line1Width) / 2);
            $line2X = (int) (($width - $line2Width) / 2);
            
            $totalTextHeight = ($fontHeight * 2) + 10;
            $startY = (int) (($height - $totalTextHeight) / 2);
            
            imagestring($image, $font, $line1X, $startY, $line1, $textColor);
            imagestring($image, $font, $line2X, $startY + $fontHeight + 10, $line2, $textColor);
            
            return;
        }

        // Calculate text dimensions
        $line1Box = imagettfbbox($fontSize, 0, $fontPath, $line1);
        $line2Box = imagettfbbox($fontSize, 0, $fontPath, $line2);
        
        $line1Width = abs($line1Box[4] - $line1Box[0]);
        $line2Width = abs($line2Box[4] - $line2Box[0]);
        $lineHeight = abs($line1Box[5] - $line1Box[1]);
        
        // Center-align text
        $line1X = (int) (($width - $line1Width) / 2);
        $line2X = (int) (($width - $line2Width) / 2);
        
        // Position in center of image
        $totalTextHeight = ($lineHeight * 2) + 10; // 10px spacing between lines
        $startY = (int) (($height - $totalTextHeight) / 2);
        
        $textY1 = $startY + $lineHeight;
        $textY2 = $textY1 + $lineHeight + 10;
        
        // Draw centered text without background
        imagettftext($image, $fontSize, 0, $line1X, $textY1, $textColor, $fontPath, $line1);
        imagettftext($image, $fontSize, 0, $line2X, $textY2, $textColor, $fontPath, $line2);
    }

    private function getSystemFont(): string
    {
        // Try common system font paths
        $fonts = [
            '/System/Library/Fonts/Helvetica.ttc',
            '/usr/share/fonts/truetype/dejavu/DejaVuSans.ttf',
            '/usr/share/fonts/truetype/liberation/LiberationSans-Regular.ttf',
            '/Windows/Fonts/arial.ttf',
            storage_path('fonts/arial.ttf'),
        ];
        
        foreach ($fonts as $font) {
            if (file_exists($font)) {
                return $font;
            }
        }
        
        // Fallback to GD built-in font (will use imagestring instead of imagettftext)
        return '';
    }
}
