<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class WhatsAppService
{
    public function sendMessage(string $to, string $message): bool
    {
        $accessToken = (string) config('services.whatsapp.access_token');
        $phoneNumberId = (string) config('services.whatsapp.phone_number_id');
        $version = (string) config('services.whatsapp.version', 'v20.0');

        if ($accessToken === '' || $phoneNumberId === '') {
            throw new \RuntimeException('WhatsApp credentials are not configured. Set WHATSAPP_ACCESS_TOKEN and WHATSAPP_PHONE_NUMBER_ID.');
        }

        $url = "https://graph.facebook.com/{$version}/{$phoneNumberId}/messages";

        $payload = [
            'messaging_product' => 'whatsapp',
            'to' => $to,
            'type' => 'text',
            'text' => [
                'body' => $message,
            ],
        ];

        $response = Http::timeout(20)
            ->retry(2, 250)
            ->withToken($accessToken)
            ->post($url, $payload);

        if (!$response->successful()) {
            Log::warning('WhatsApp API request failed', [
                'status' => $response->status(),
                'body' => $response->json(),
            ]);

            return false;
        }

        return true;
    }

    public function sendTemplate(string $to, string $templateName, string $languageCode = 'en_US', array $components = []): bool
    {
        $accessToken = (string) config('services.whatsapp.access_token');
        $phoneNumberId = (string) config('services.whatsapp.phone_number_id');
        $version = (string) config('services.whatsapp.version', 'v20.0');

        if ($accessToken === '' || $phoneNumberId === '') {
            throw new \RuntimeException('WhatsApp credentials are not configured. Set WHATSAPP_ACCESS_TOKEN and WHATSAPP_PHONE_NUMBER_ID.');
        }

        $url = "https://graph.facebook.com/{$version}/{$phoneNumberId}/messages";

        $template = [
            'name' => $templateName,
            'language' => ['code' => $languageCode],
        ];

        if (!empty($components)) {
            $template['components'] = $components;
        }

        $payload = [
            'messaging_product' => 'whatsapp',
            'to' => $to,
            'type' => 'template',
            'template' => $template,
        ];

        $response = Http::timeout(20)
            ->retry(2, 250)
            ->withToken($accessToken)
            ->post($url, $payload);

        if (!$response->successful()) {
            Log::warning('WhatsApp API template request failed', [
                'status' => $response->status(),
                'body' => $response->json(),
            ]);

            return false;
        }

        return true;
    }

    public function sendOtp(string $to, string $otp, int $expiresInSeconds): bool
    {
        $templateName = (string) config('services.whatsapp.otp_template_name');
        $languageCode = (string) config('services.whatsapp.otp_template_language', 'en_US');
        $paramsRaw = (string) config('services.whatsapp.otp_template_params', 'otp,minutes');
        $minutes = (string) (int) ceil($expiresInSeconds / 60);

        if ($templateName !== '') {
            $normalizedParams = strtolower(trim($paramsRaw));
            if ($normalizedParams === '' || $normalizedParams === 'none' || $normalizedParams === '0' || $normalizedParams === 'null') {
                return $this->sendTemplate($to, $templateName, $languageCode);
            }

            $paramKeys = array_values(array_filter(array_map('trim', explode(',', $paramsRaw)), fn ($v) => $v !== ''));
            if (empty($paramKeys)) {
                return $this->sendTemplate($to, $templateName, $languageCode);
            }

            $parameters = [];
            foreach ($paramKeys as $key) {
                $value = match ($key) {
                    'otp' => $otp,
                    'minutes' => $minutes,
                    'app_name' => (string) config('app.name'),
                    default => null,
                };

                if ($value !== null) {
                    $parameters[] = ['type' => 'text', 'text' => (string) $value];
                }
            }

            $components = [[
                'type' => 'body',
                'parameters' => $parameters,
            ]];

            return $this->sendTemplate($to, $templateName, $languageCode, $components);
        }

        $message = 'Your ' . config('app.name') . ' verification code is ' . $otp . '. It expires in ' . (int) $minutes . ' minutes.';

        return $this->sendMessage($to, $message);
    }
}
