# Email Configuration Guide

## Overview

The application sends professional welcome emails with email verification links when users register. This guide explains how to configure email sending for different environments.

## Email Features

### Welcome Email
- **Professional Design**: Beautiful, responsive HTML email template
- **Email Verification**: Secure, time-limited verification link (60 minutes)
- **Platform Features**: Highlights key features of the platform
- **Security Notice**: Informs users about account security
- **Retry Logic**: Automatic retry on failure (3 attempts with 60-second backoff)

## Configuration

### Environment Variables

Update your `.env` file with the appropriate mail configuration:

```env
# Mail Configuration
MAIL_MAILER=smtp
MAIL_HOST=your-smtp-host.com
MAIL_PORT=587
MAIL_USERNAME=your-email@domain.com
MAIL_PASSWORD=your-email-password
MAIL_ENCRYPTION=tls
MAIL_FROM_ADDRESS=noreply@yourdomain.com
MAIL_FROM_NAME="${APP_NAME}"
```

### Common Mail Providers

#### Gmail
```env
MAIL_MAILER=smtp
MAIL_HOST=smtp.gmail.com
MAIL_PORT=587
MAIL_USERNAME=your-gmail@gmail.com
MAIL_PASSWORD=your-app-password
MAIL_ENCRYPTION=tls
MAIL_FROM_ADDRESS=your-gmail@gmail.com
MAIL_FROM_NAME="${APP_NAME}"
```

**Note**: For Gmail, you need to use an [App Password](https://support.google.com/accounts/answer/185833), not your regular password.

#### SendGrid
```env
MAIL_MAILER=smtp
MAIL_HOST=smtp.sendgrid.net
MAIL_PORT=587
MAIL_USERNAME=apikey
MAIL_PASSWORD=your-sendgrid-api-key
MAIL_ENCRYPTION=tls
MAIL_FROM_ADDRESS=noreply@yourdomain.com
MAIL_FROM_NAME="${APP_NAME}"
```

#### Mailgun
```env
MAIL_MAILER=mailgun
MAILGUN_DOMAIN=your-domain.mailgun.org
MAILGUN_SECRET=your-mailgun-api-key
MAIL_FROM_ADDRESS=noreply@yourdomain.com
MAIL_FROM_NAME="${APP_NAME}"
```

#### Amazon SES
```env
MAIL_MAILER=ses
AWS_ACCESS_KEY_ID=your-access-key
AWS_SECRET_ACCESS_KEY=your-secret-key
AWS_DEFAULT_REGION=us-east-1
MAIL_FROM_ADDRESS=noreply@yourdomain.com
MAIL_FROM_NAME="${APP_NAME}"
```

#### Mailtrap (Development/Testing)
```env
MAIL_MAILER=smtp
MAIL_HOST=smtp.mailtrap.io
MAIL_PORT=2525
MAIL_USERNAME=your-mailtrap-username
MAIL_PASSWORD=your-mailtrap-password
MAIL_ENCRYPTION=tls
MAIL_FROM_ADDRESS=test@example.com
MAIL_FROM_NAME="${APP_NAME}"
```

### Development Environment

For local development, you have several options:

#### Option 1: Log Driver (No actual emails sent)
```env
MAIL_MAILER=log
```
Emails will be written to `storage/logs/laravel.log` instead of being sent.

#### Option 2: Mailtrap (Recommended for testing)
Sign up at [mailtrap.io](https://mailtrap.io) and use the credentials provided.

#### Option 3: MailHog (Local SMTP server)
```bash
# Install MailHog
brew install mailhog  # macOS
# or download from https://github.com/mailhog/MailHog

# Run MailHog
mailhog
```

```env
MAIL_MAILER=smtp
MAIL_HOST=127.0.0.1
MAIL_PORT=1025
MAIL_USERNAME=null
MAIL_PASSWORD=null
MAIL_ENCRYPTION=null
```

Access MailHog UI at: http://localhost:8025

## Queue Configuration

Email sending is handled asynchronously using Laravel queues to prevent blocking user registration.

### Queue Setup

1. **Ensure queue table exists:**
```bash
php artisan queue:table
php artisan migrate
```

2. **Configure queue connection in `.env`:**
```env
QUEUE_CONNECTION=database
```

3. **Run the queue worker:**
```bash
php artisan queue:work
```

For production, use a process manager like Supervisor to keep the queue worker running.

### Supervisor Configuration (Production)

Create `/etc/supervisor/conf.d/laravel-worker.conf`:

```ini
[program:laravel-worker]
process_name=%(program_name)s_%(process_num)02d
command=php /path/to/your/app/artisan queue:work --sleep=3 --tries=3 --max-time=3600
autostart=true
autorestart=true
stopasgroup=true
killasgroup=true
user=www-data
numprocs=2
redirect_stderr=true
stdout_logfile=/path/to/your/app/storage/logs/worker.log
stopwaitsecs=3600
```

Then:
```bash
sudo supervisorctl reread
sudo supervisorctl update
sudo supervisorctl start laravel-worker:*
```

## Email Template Customization

### Template Location
The welcome email template is located at:
```
resources/views/emails/welcome.blade.php
```

### Customization Options

1. **Update branding colors:**
   - Edit the gradient colors in the CSS (currently purple gradient)
   - Update button styles
   - Modify header background

2. **Add/remove features:**
   - Edit the features section in the template
   - Add your platform-specific features

3. **Update content:**
   - Modify greeting messages
   - Update security notices
   - Change footer links

### Email Preview

To preview the email without sending:

```php
// In tinker or a test route
use App\Models\User;
use App\Mail\WelcomeEmail;

$user = User::first();
return new WelcomeEmail($user);
```

## Email Verification Flow

### 1. User Registration
- User registers with email and password
- Welcome email is queued for sending
- User receives JWT tokens immediately

### 2. Email Verification
- User clicks "Verify Email Address" button in email
- Verification link is valid for 60 minutes
- Link format: `/api/v1/auth/verify-email/{id}/{hash}`

### 3. Verification Endpoint
```
GET /api/v1/auth/verify-email/{id}/{hash}
```

**Response (Success):**
```json
{
  "success": true,
  "message": "Email verified successfully.",
  "data": {
    "user": { ... }
  }
}
```

### 4. Resend Verification Email
```
POST /api/v1/auth/resend-verification
Authorization: Bearer {token}
```

**Rate Limited:** 3 requests per hour

## Troubleshooting

### Emails Not Sending

1. **Check queue worker is running:**
```bash
php artisan queue:work
```

2. **Check failed jobs:**
```bash
php artisan queue:failed
```

3. **Retry failed jobs:**
```bash
php artisan queue:retry all
```

4. **Check logs:**
```bash
tail -f storage/logs/laravel.log
```

### SMTP Connection Issues

1. **Verify credentials:**
   - Double-check MAIL_USERNAME and MAIL_PASSWORD
   - Ensure MAIL_HOST and MAIL_PORT are correct

2. **Check firewall:**
   - Ensure outbound connections on SMTP port are allowed
   - Common ports: 587 (TLS), 465 (SSL), 25 (unencrypted)

3. **Test connection:**
```bash
telnet smtp.example.com 587
```

### Gmail "Less Secure Apps" Error

Gmail no longer supports "less secure apps". You must use an App Password:
1. Enable 2-Factor Authentication on your Google account
2. Generate an App Password: https://myaccount.google.com/apppasswords
3. Use the generated password in MAIL_PASSWORD

### Email Delivery Issues

1. **Check spam folder:**
   - Emails might be marked as spam
   - Configure SPF, DKIM, and DMARC records for your domain

2. **Verify sender domain:**
   - Use a verified domain in MAIL_FROM_ADDRESS
   - Avoid using @gmail.com or @yahoo.com for production

3. **Email reputation:**
   - Use a dedicated email service (SendGrid, Mailgun, SES)
   - Warm up your sending domain gradually

## Production Best Practices

1. **Use a dedicated email service:**
   - SendGrid, Mailgun, Amazon SES, Postmark
   - Better deliverability and analytics

2. **Configure DNS records:**
   - SPF: Specify authorized mail servers
   - DKIM: Email authentication
   - DMARC: Email validation policy

3. **Monitor email delivery:**
   - Track bounce rates
   - Monitor spam complaints
   - Review delivery analytics

4. **Queue workers:**
   - Use Supervisor or systemd to manage queue workers
   - Run multiple workers for high volume
   - Monitor worker health

5. **Rate limiting:**
   - Respect email provider limits
   - Implement exponential backoff for retries
   - Use queue delays for bulk emails

## Testing

### Manual Testing

```bash
# Send test email via tinker
php artisan tinker

use App\Jobs\SendWelcomeEmail;
use App\Models\User;

$user = User::first();
SendWelcomeEmail::dispatch($user);
```

### Automated Testing

```php
use App\Mail\WelcomeEmail;
use Illuminate\Support\Facades\Mail;

public function test_welcome_email_is_sent_on_registration()
{
    Mail::fake();

    $response = $this->postJson('/api/v1/auth/register', [
        'full_name' => 'Test User',
        'email' => 'test@example.com',
        'password' => 'Password@123',
        'password_confirmation' => 'Password@123',
    ]);

    Mail::assertQueued(WelcomeEmail::class, function ($mail) {
        return $mail->user->email === 'test@example.com';
    });
}
```

## Support

For issues or questions:
- Check Laravel Mail documentation: https://laravel.com/docs/mail
- Review queue documentation: https://laravel.com/docs/queues
- Check application logs: `storage/logs/laravel.log`
