# Security Architecture

## Overview

This document outlines the security measures implemented in the Yilick API authentication system.

## Authentication Security

### JWT Token Security

1. **Token Signing**: Uses HS256 algorithm with a strong secret key
2. **Token Expiration**: Access tokens expire after 60 minutes (configurable)
3. **Refresh Tokens**: Separate refresh tokens with 14-day expiration
4. **Token Blacklisting**: Revoked tokens are blacklisted to prevent reuse
5. **Custom Claims**: Tokens include user role and status for authorization

### Refresh Token Implementation

- Refresh tokens are stored as SHA-256 hashes (never plain text)
- Each refresh token is tied to a specific access token (via JTI)
- Tokens track IP address and user agent for audit purposes
- Refresh token rotation: old token is revoked when new one is issued

### Password Security

1. **Hashing**: Bcrypt with 12 rounds (configurable via `BCRYPT_ROUNDS`)
2. **Validation Rules**:
   - Minimum 8 characters
   - Mixed case required
   - Numbers required
   - Special characters required
   - Checked against known breached passwords (Have I Been Pwned)
3. **Password Reset**: Time-limited tokens (60 minutes), single-use

## Rate Limiting

### Endpoint-Specific Limits

| Endpoint Category | Limit | Purpose |
|------------------|-------|---------|
| Login/Register | 5/min per IP | Prevent brute force attacks |
| OTP Requests | 3/min per identifier | Prevent OTP flooding |
| Email Verification | 1/min per user | Prevent email spam |
| General API | 60/min per user/IP | Prevent API abuse |

### Implementation

- Uses Laravel's built-in rate limiter with Redis backend
- Returns `429 Too Many Requests` with `Retry-After` header
- Rate limits are applied per IP for unauthenticated requests
- Rate limits are applied per user ID for authenticated requests

## Input Validation & Sanitization

### Form Request Validation

All inputs are validated using Laravel Form Requests:

- **Email**: RFC-compliant validation with DNS check
- **Phone**: E.164 format validation (`/^\+?[1-9]\d{1,14}$/`)
- **File Uploads**: MIME type, size, and dimension validation
- **Dates**: Range validation (after 1900, before today)

### SQL Injection Prevention

- All database queries use Eloquent ORM with parameterized queries
- No raw SQL queries with user input

### XSS Prevention

- API-only architecture (no HTML rendering)
- JSON responses with proper Content-Type headers
- Input sanitization through validation rules

## File Upload Security

### Profile Photo Validation

```php
'profile_photo' => [
    'image',
    'mimes:jpeg,jpg,png,gif,webp',
    'max:5120',  // 5MB
    'dimensions:min_width=100,min_height=100,max_width=4096,max_height=4096',
]
```

### Storage Security

- Files stored in `storage/app/public/profile-photos`
- Unique filenames generated to prevent overwrites
- Old files deleted when replaced
- Cloud storage ready (S3, etc.)

## Account Security

### Account Status Management

- **Active**: Normal account access
- **Suspended**: Login blocked, existing tokens revoked
- **Deleted**: Soft-deleted, can be restored by admin

### Session Management

- Logout invalidates current token
- "Logout All" revokes all refresh tokens
- Password change revokes other sessions
- Account suspension revokes all tokens

## Authorization

### Role-Based Access Control (RBAC)

| Role | Permissions |
|------|-------------|
| User | Own profile management |
| Seller | User permissions + listing management (future) |
| Admin | Full system access, user management |

### Policy-Based Authorization

- `UserPolicy` controls access to user resources
- Admins cannot modify their own role/status
- Users can only access their own data

## Security Headers

All API responses include:

```
X-Content-Type-Options: nosniff
X-Frame-Options: DENY
X-XSS-Protection: 1; mode=block
Strict-Transport-Security: max-age=31536000; includeSubDomains
Content-Security-Policy: default-src 'none'; frame-ancestors 'none'
Referrer-Policy: strict-origin-when-cross-origin
Permissions-Policy: geolocation=(), microphone=(), camera=()
Cache-Control: no-store, no-cache, must-revalidate, proxy-revalidate
Pragma: no-cache
```

## CSRF Protection

- CSRF protection is disabled for API routes (stateless JWT auth)
- API routes use token-based authentication instead
- CORS configured to restrict origins in production

## OTP Security

### Implementation

- 6-digit numeric codes
- SHA-256 hashed storage
- 10-minute expiration
- Maximum 5 verification attempts
- Rate limited to 3 requests per minute

### Best Practices

- OTP is only returned in debug mode (development)
- Production should integrate with SMS/Email providers
- Failed attempts are tracked and limited

## Audit & Logging

### Tracked Events

- User registration
- Login attempts (success/failure)
- Password changes
- Token refresh
- Account status changes
- Admin actions

### Token Metadata

Refresh tokens store:
- User ID
- Token ID (JTI)
- IP address
- User agent
- Creation time
- Expiration time
- Revocation time

## Production Recommendations

### Environment Configuration

```env
APP_ENV=production
APP_DEBUG=false
JWT_SECRET=<strong-random-key>
JWT_TTL=60
JWT_REFRESH_TTL=20160
BCRYPT_ROUNDS=12
```

### Additional Measures

1. **HTTPS Only**: Enforce HTTPS in production
2. **Database Encryption**: Encrypt sensitive fields at rest
3. **Log Monitoring**: Set up alerts for suspicious activity
4. **Regular Audits**: Review access logs and token usage
5. **Backup Strategy**: Regular encrypted backups
6. **Key Rotation**: Periodic JWT secret rotation with grace period

### Infrastructure

1. **WAF**: Web Application Firewall for additional protection
2. **DDoS Protection**: CloudFlare or similar service
3. **Database Security**: Separate credentials, network isolation
4. **Redis Security**: Password protection, network isolation

## Compliance Considerations

This implementation supports compliance with:

- **GDPR**: Soft delete, data export capability
- **PCI-DSS**: Secure password handling, audit logging
- **OWASP Top 10**: Protection against common vulnerabilities

## Incident Response

### Token Compromise

1. Revoke all user tokens via admin endpoint
2. Force password reset
3. Review audit logs
4. Notify affected user

### Mass Breach

1. Rotate JWT secret
2. Invalidate all tokens (clear blacklist cache)
3. Force password reset for all users
4. Review and patch vulnerability
