@extends('layouts.app')

@section('title', $listing->title . ' | Buy on Yilick Uganda')
@section('meta_description', Str::limit(strip_tags($listing->description), 160))
@section('meta_keywords', $listing->category . ', buy ' . $listing->title . ' uganda, safe shopping, yilick marketplace')
@section('og_image', $listing->media->first()->url ?? asset('assets/icons/listing_placeholder.svg'))

@section('header_meta')
<script type="application/ld+json">
{
    "@context": "https://schema.org/",
    "@type": "Product",
    "name": "{{ $listing->title }}",
    "image": [
        @foreach($listing->media as $media)
            "{{ $media->url }}"{{ !$loop->last ? ',' : '' }}
        @endforeach
    ],
    "description": "{{ Str::limit(strip_tags($listing->description), 200) }}",
    "sku": "YL-{{ $listing->id }}",
    "mpn": "{{ $listing->id }}",
    "brand": {
        "@type": "Brand",
        "name": "{{ $listing->business->name ?: 'Yilick Seller' }}"
    },
    "aggregateRating": {
        "@type": "AggregateRating",
        "ratingValue": "5.0",
        "reviewCount": "{{ max(1, $listing->business->feedbacks_count ?? 0) }}"
    },
    "offers": {
        "@type": "Offer",
        "url": "{{ url()->current() }}",
        "priceCurrency": "UGX",
        "price": "{{ $listing->price_amount }}",
        "priceValidUntil": "{{ now()->addMonths(6)->format('Y-12-31') }}",
        @php
            $conditionInput = strtolower($listing->custom_fields['Condition'] ?? $listing->custom_fields['condition'] ?? 'new');
            $condition = 'https://schema.org/NewCondition';
            if (str_contains($conditionInput, 'used')) $condition = 'https://schema.org/UsedCondition';
            elseif (str_contains($conditionInput, 'refurbished')) $condition = 'https://schema.org/RefurbishedCondition';
            elseif (str_contains($conditionInput, 'damaged')) $condition = 'https://schema.org/DamagedCondition';
        @endphp
        "itemCondition": "{{ $condition }}",
        "availability": "https://schema.org/InStock",
        "hasMerchantReturnPolicy": {
            "@type": "MerchantReturnPolicy",
            "applicableCountry": "UG",
            "returnPolicyCategory": "https://schema.org/MerchantReturnFiniteReturnPeriod",
            "merchantReturnDays": "7",
            "returnMethod": "https://schema.org/ReturnByMail",
            "returnFees": "https://schema.org/ReturnFeesCustomerResponsibility"
        },
        "shippingDetails": {
            "@type": "OfferShippingDetails",
            "shippingRate": {
                "@type": "MonetaryAmount",
                "value": "0",
                "currency": "UGX"
            },
            "deliveryTime": {
                "@type": "ShippingDeliveryTime",
                "handlingTime": {
                    "@type": "QuantitativeValue",
                    "minValue": "0",
                    "maxValue": "1",
                    "unitCode": "d"
                },
                "transitTime": {
                    "@type": "QuantitativeValue",
                    "minValue": "1",
                    "maxValue": "5",
                    "unitCode": "d"
                }
            },
            "shippingDestination": {
                "@type": "DefinedRegion",
                "addressCountry": "UG"
            }
        },
        "seller": {
            "@type": "Organization",
            "name": "{{ $listing->business->name }}"
        }
    }
},
{
    "@context": "https://schema.org",
    "@type": "BreadcrumbList",
    "itemListElement": [{
        "@type": "ListItem",
        "position": 1,
        "name": "Home",
        "item": "{{ route('web.home') }}"
    },{
        "@type": "ListItem",
        "position": 2,
        "name": "{{ $listing->category ?: 'Listings' }}",
        "item": "{{ route('listing.search', ['category' => $listing->category]) }}"
    },{
        "@type": "ListItem",
        "position": 3,
        "name": "{{ $listing->title }}",
        "item": "{{ url()->current() }}"
    }]
}
</script>
@endsection

@section('content')
<div class="bg-gray-50 min-h-screen pb-12">
    <!-- Breadcrumbs -->
    <div class="bg-white border-b border-gray-200">
        <div class="container mx-auto px-4 py-3">
            <nav class="flex text-sm text-gray-500 overflow-x-auto whitespace-nowrap">
                <a href="/" class="hover:text-primary transition-colors">Home</a>
                <span class="mx-2">/</span>
                <span class="font-medium text-gray-900 truncate">{{ Str::limit($listing->title, 40) }}</span>
            </nav>
        </div>
    </div>

    <div class="container mx-auto px-4 py-6">
        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6 lg:gap-8">
            
            <!-- Main Content (Left Column) -->
            <div class="lg:col-span-2 space-y-6">
                
                <!-- Image Gallery -->
                <div class="bg-white rounded-2xl shadow-sm border border-gray-100 overflow-hidden relative group select-none" id="gallery-container">
                    <div class="aspect-[4/3] bg-gray-100 relative overflow-hidden group/image">
                        @if($listing->media->isNotEmpty())
                            <!-- Blurred Background for aesthetics -->
                            <img src="{{ $listing->media->first()->url }}" 
                                 id="gallery-bg"
                                 onerror="this.onerror=null; this.src='/assets/icons/listing_placeholder.svg'"
                                 class="absolute inset-0 w-full h-full object-cover blur-2xl opacity-40 scale-110 transition-all duration-500">
                            
                            <!-- Main Image -->
                            <img src="{{ $listing->media->first()->url }}" 
                                 alt="{{ $listing->title }}" 
                                 id="main-image"
                                 onerror="this.onerror=null; this.src='/assets/icons/listing_placeholder.svg'"
                                 onclick="AdGallery.openLightbox()"
                                 class="relative z-10 w-full h-full object-contain transition-transform duration-300 cursor-pointer lg:hover:scale-105 active:scale-95">

                            <!-- Navigation Arrows (Hidden if single image) -->
                            @if($listing->media->count() > 1)
                                <button onclick="AdGallery.prev()" class="absolute left-4 top-1/2 -translate-y-1/2 w-10 h-10 rounded-full bg-white/90 hover:bg-white text-gray-800 shadow-lg flex items-center justify-center opacity-0 group-hover/image:opacity-100 transition-all z-20 hover:scale-110 active:scale-95">
                                    <i class="fa-solid fa-chevron-left"></i>
                                </button>
                                <button onclick="AdGallery.next()" class="absolute right-4 top-1/2 -translate-y-1/2 w-10 h-10 rounded-full bg-white/90 hover:bg-white text-gray-800 shadow-lg flex items-center justify-center opacity-0 group-hover/image:opacity-100 transition-all z-20 hover:scale-110 active:scale-95">
                                    <i class="fa-solid fa-chevron-right"></i>
                                </button>
                            @endif

                            <!-- Image Counter -->
                            <div class="absolute bottom-4 left-4 z-20 bg-black/60 backdrop-blur-md px-3 py-1.5 rounded-full flex items-center gap-2">
                                <i class="fa-solid fa-camera text-white/80 text-xs"></i>
                                <span class="text-xs font-bold text-white tracking-wider">
                                    <span id="current-index">1</span> / {{ $listing->media->count() }}
                                </span>
                            </div>
                            
                            <!-- Actions -->
                            <div class="absolute top-4 right-4 z-20 flex gap-2">
                                <button 
                                    type="button"
                                    onclick="toggleFavoriteGlobal(this, 'listing', {{ $listing->id }})"
                                    data-active="{{ Auth::guard('web')->user() && $listing->isFavoritedBy(Auth::guard('web')->user()) ? 'true' : 'false' }}"
                                    class="w-10 h-10 rounded-full bg-white/90 hover:bg-white {{ Auth::guard('web')->user() && $listing->isFavoritedBy(Auth::guard('web')->user()) ? 'text-red-500' : 'text-gray-700 hover:text-red-500' }} shadow-md flex items-center justify-center transition-all hover:scale-105 active:scale-95 favorite-btn">
                                    <i class="{{ Auth::guard('web')->user() && $listing->isFavoritedBy(Auth::guard('web')->user()) ? 'fa-solid' : 'fa-regular' }} fa-heart text-lg"></i>
                                </button>
                                <button onclick="shareAd('{{ addslashes($listing->title) }}', '{{ route('listing.share', $listing->slug) }}')" class="w-10 h-10 rounded-full bg-white/90 hover:bg-white text-gray-700 hover:text-primary shadow-md flex items-center justify-center transition-all hover:scale-105 active:scale-95">
                                    <i class="fa-solid fa-share-nodes text-lg"></i>
                                </button>
                            </div>
                        @else
                            <div class="w-full h-full flex items-center justify-center text-gray-300 bg-gray-50">
                                <i class="fa-solid fa-image text-6xl opacity-50"></i>
                            </div>
                        @endif
                    </div>

                    <!-- Thumbnails -->
                    @if($listing->media->count() > 1)
                    <div class="p-3 border-t border-gray-100 bg-white overflow-x-auto custom-scrollbar">
                        <div class="flex gap-2.5">
                            @foreach($listing->media as $index => $media)
                            <button onclick="AdGallery.setIndex({{ $index }})" 
                                    id="thumb-{{ $index }}"
                                    class="w-16 h-16 flex-shrink-0 rounded-lg border-2 {{ $loop->first ? 'border-primary ring-2 ring-primary/20' : 'border-transparent opacity-70 hover:opacity-100' }} overflow-hidden transition-all relative">
                                <img src="{{ $media->url }}" onerror="this.onerror=null; this.src='/assets/icons/listing_placeholder.svg'" class="w-full h-full object-cover">
                            </button>
                            @endforeach
                        </div>
                    </div>
                    @endif
                </div>

                <!-- Description & Details (Mobile: Below Gallery) -->
                <div class="bg-white rounded-2xl shadow-sm border border-gray-100 p-6 md:p-8">
                    <div class="flex items-start justify-between mb-6">
                        <div>
                            <div class="flex items-center gap-2 text-sm text-gray-500 mb-2">
                                <span class="bg-gray-100 px-2 py-0.5 rounded text-xs font-semibold text-gray-600 uppercase">AVAILABLE</span>
                                <span>•</span>
                                <span>{{ $listing->created_at->diffForHumans() }}</span>
                                <span>•</span>
                                <div class="flex items-center gap-1">
                                    <i class="fa-solid fa-location-dot text-gray-400"></i>
                                    <span>{{ $listing->location_district ?? 'Kampala' }}</span>
                                </div>
                            </div>
                            <h1 class="text-2xl md:text-2xl font-bold text-gray-900 leading-tight mb-2">{{ $listing->title }}</h1>
                            <div class="mt-3 flex items-baseline gap-1.5">
                                <span class="text-xl md:text-2xl font-bold text-gray-500">{{ $listing->currency }}</span>
                                <span class="text-3xl md:text-5xl font-black text-primary tracking-tight">{{ number_format($listing->price_amount) }}</span>
                            </div>
                        </div>
                    </div>

                    <hr class="border-gray-100 my-6">

                    <h3 class="text-lg font-bold text-gray-900 mb-4">Description</h3>
                    @if(Str::length($listing->description) > 300)
                        <div class="relative">
                            <div id="description-content" class="prose prose-green max-w-none text-gray-600 leading-relaxed max-h-32 overflow-hidden transition-all duration-500">
                                {!! nl2br(e($listing->description)) !!}
                            </div>
                            <div id="description-gradient" class="absolute bottom-0 left-0 right-0 h-24 bg-gradient-to-t from-white via-white/80 to-transparent"></div>
                        </div>
                        <button id="description-toggle" onclick="toggleDescription()" class="mt-3 text-primary font-semibold hover:text-green-700 flex items-center gap-2 text-sm transition-colors">
                            <span id="description-toggle-text">Read more</span>
                            <i id="description-toggle-icon" class="fa-solid fa-chevron-down transition-transform duration-300"></i>
                        </button>
                    @else
                        <div class="prose prose-green max-w-none text-gray-600 leading-relaxed">
                            {!! nl2br(e($listing->description)) !!}
                        </div>
                    @endif

                    @if(!empty($listing->attributes) || !empty($listing->custom_fields))
                    <hr class="border-gray-100 my-6">
                    <h3 class="text-lg font-bold text-gray-900 mb-4">Details</h3>
                    <div class="grid grid-cols-2 md:grid-cols-3 gap-y-4 gap-x-8">
                        @if($listing->attributes)
                            @foreach($listing->attributes as $key => $value)
                            <div>
                                <p class="text-xs text-uppercase text-gray-400 font-semibold mb-1">{{ Str::title(str_replace('_', ' ', $key)) }}</p>
                                <p class="text-sm font-medium text-gray-800">{{ $value }}</p>
                            </div>
                            @endforeach
                        @endif

                        @if($listing->custom_fields)
                            @foreach($listing->custom_fields as $key => $value)
                            <div>
                                <p class="text-xs text-uppercase text-gray-400 font-semibold mb-1">{{ Str::title(str_replace('_', ' ', $key)) }}</p>
                                <p class="text-sm font-medium text-gray-800">{{ is_array($value) ? implode(', ', $value) : $value }}</p>
                            </div>
                            @endforeach
                        @endif
                    </div>
                    @endif
                </div>

            </div>

            <!-- Sidebar (Right Column - Sticky) -->
            <div class="lg:col-span-1 space-y-6">
                @if(auth('web')->check() && auth('web')->id() == $listing->created_by_user_id)
                <!-- Owner Management Card -->
                <div class="bg-white rounded-2xl shadow-sm border border-emerald-100 p-6 relative overflow-hidden group">
                    <!-- Decorative background -->
                    <div class="absolute top-0 right-0 -translate-y-1/2 translate-x-1/2 w-32 h-32 bg-emerald-50 rounded-full opacity-50 group-hover:scale-110 transition-transform duration-500"></div>
                    
                    <div class="relative">
                        <div class="flex items-center gap-3 mb-6">
                            <div class="w-10 h-10 rounded-xl bg-emerald-50 text-emerald-600 flex items-center justify-center">
                                <i class="fa-solid fa-user-gear"></i>
                            </div>
                            <div>
                                <h3 class="text-sm font-black text-gray-900 uppercase tracking-tight">Owner Management</h3>
                                <p class="text-[10px] text-emerald-600 font-bold">You posted this ad</p>
                            </div>
                        </div>

                        <div class="grid grid-cols-1 gap-3">
                            <a href="{{ route('listing.edit', $listing->slug) }}" class="w-full bg-white border border-gray-200 hover:border-emerald-500 hover:text-emerald-600 text-gray-700 font-bold py-3 rounded-xl transition-all flex items-center justify-center gap-2 group/btn">
                                <i class="fa-regular fa-pen-to-square group-hover/btn:scale-110 transition-transform text-emerald-500"></i>
                                <span>Edit Ad Details</span>
                            </a>

                            @if($listing->status === \App\Models\Listing::STATUS_PUBLISHED)
                                <button onclick="handleAdAction('close', {{ $listing->id }})" class="w-full bg-gray-50 border border-gray-100 hover:bg-orange-50 hover:border-orange-200 hover:text-orange-600 text-gray-600 font-bold py-3 rounded-xl transition-all flex items-center justify-center gap-2 group/btn">
                                    <i class="fa-solid fa-circle-xmark group-hover/btn:rotate-12 transition-transform text-orange-400"></i>
                                    <span>Close Ad</span>
                                </button>
                            @endif

                            <button onclick="handleAdAction('delete', {{ $listing->id }})" class="w-full bg-red-50/30 border border-red-100 hover:bg-red-50 hover:border-red-200 hover:text-red-600 text-red-500 font-bold py-3 rounded-xl transition-all flex items-center justify-center gap-2 group/btn">
                                <i class="fa-regular fa-trash-can group-hover/btn:shake transition-transform"></i>
                                <span>Delete Ad</span>
                            </button>
                        </div>
                    </div>
                </div>
                @endif
                
                <!-- Seller Card -->
                <!-- Seller Card -->
                <div class="bg-white rounded-2xl shadow-sm border border-gray-100 p-6 sticky top-24">
                    @php
                        // Unified Business Logic to match API structure
                        // If business exists, use it. Else map createdBy (User) to business-like structure.
                        $seller = $listing->business ? (object)[
                            'name' => $listing->business->name,
                            'avatar' => $listing->business->logo_url,
                            'verified' => $listing->business->verification_status === 'verified',
                            'member_since' => $listing->business->created_at,
                            'rating' => 5.0, // Default or fetch real
                            'has_store' => true,
                            'slug' => $listing->business->slug,
                            'phone' => $listing->business->primary_phone,
                            'whatsapp' => $listing->business->primary_whatsapp,
                        ] : (object)[
                            'name' => $listing->createdBy?->name ?: 'Yilick User',
                            // Fallback to User Profile Photo. If null, blade logic below handles initial.
                            'avatar' => $listing->createdBy?->profile_photo_url, 
                            'verified' => $listing->createdBy?->verified,
                            'member_since' => $listing->createdBy?->created_at,
                            'rating' => 5.0,
                            'has_store' => false,
                            'slug' => null,
                            'phone' => $listing->createdBy?->phone,
                            'whatsapp' => $listing->createdBy?->whatsapp_number,
                        ];
                    @endphp

                    <!-- Seller Info (Avatar/Name) -->
                    <div class="flex items-center gap-4 mb-6">
                        <div class="w-16 h-16 rounded-full bg-gray-100 flex items-center justify-center overflow-hidden border border-gray-200">
                             @if(!empty($seller->avatar))
                                <img src="{{ $seller->avatar }}" onerror="this.src='{{ asset('assets/icons/favicon.png') }}'" class="w-full h-full object-cover">
                             @else
                                <span class="text-2xl font-bold text-gray-400">{{ substr($seller->name, 0, 1) }}</span>
                             @endif
                        </div>
                        <div>
                            <h3 class="text-lg font-bold text-gray-900 flex items-center gap-1">
                                {{ $seller->name }}
                                @if($seller->verified)
                                    <i class="fa-solid fa-circle-check text-blue-500 text-sm" title="Verified Seller"></i>
                                @endif
                            </h3>
                            <p class="text-sm text-gray-500">Member since {{ optional($seller->member_since)->format('M Y') ?? 'N/A' }}</p>
                            <div class="flex items-center gap-1 mt-1 text-xs font-semibold text-yellow-500">
                                <i class="fa-solid fa-star"></i>
                                <span>{{ $seller->rating }}</span>
                                <span class="text-gray-400 font-normal">(0 reviews)</span>
                            </div>
                        </div>
                    </div>

                    <!-- Visit Store Button (New) -->
                    @if($seller->has_store)
                    <a href="{{ route('business.show', $seller->slug) }}" class="block w-full text-center border border-gray-200 hover:border-primary text-gray-700 hover:text-primary font-semibold py-2 rounded-lg mb-6 transition-colors text-sm">
                        View Store Profile
                    </a>
                    @endif

                    <!-- Quick Questions -->
                    @if(auth('web')->id() != $listing->created_by_user_id)
                    <div class="mb-4">
                        <p class="text-xs font-bold text-gray-400 uppercase mb-2">Quick questions</p>
                        <div class="flex flex-wrap gap-2">
                             @auth('web')
                                <a href="{{ route('web.chat.start', ['listing' => $listing->id, 'message' => 'Is it available?']) }}" class="px-3 py-1.5 bg-gray-50 hover:bg-gray-100 text-xs text-gray-600 rounded-full border border-gray-200 transition-colors">Is it available?</a>
                                <a href="{{ route('web.chat.start', ['listing' => $listing->id, 'message' => 'What is the last price?']) }}" class="px-3 py-1.5 bg-gray-50 hover:bg-gray-100 text-xs text-gray-600 rounded-full border border-gray-200 transition-colors">Last price?</a>
                                <a href="{{ route('web.chat.start', ['listing' => $listing->id, 'message' => ' where are you located?']) }}" class="px-3 py-1.5 bg-gray-50 hover:bg-gray-100 text-xs text-gray-600 rounded-full border border-gray-200 transition-colors">Location?</a>
                             @else
                                <a href="{{ route('login') }}" class="px-3 py-1.5 bg-gray-50 hover:bg-gray-100 text-xs text-gray-600 rounded-full border border-gray-200 transition-colors">Is it available?</a>
                                <a href="{{ route('login') }}" class="px-3 py-1.5 bg-gray-50 hover:bg-gray-100 text-xs text-gray-600 rounded-full border border-gray-200 transition-colors">Last price?</a>
                                <a href="{{ route('login') }}" class="px-3 py-1.5 bg-gray-50 hover:bg-gray-100 text-xs text-gray-600 rounded-full border border-gray-200 transition-colors">Location?</a>
                             @endauth
                        </div>
                    </div>
                    @endif

                    <div class="space-y-3">
                        @if(!empty($seller->whatsapp) && auth('web')->id() != $listing->created_by_user_id)
                        <a href="https://wa.me/{{ preg_replace('/[^0-9]/', '', $seller->whatsapp) }}?text=Hi, I am interested in your ad: {{ urlencode($listing->title) }} {{ urlencode(route('listing.share', $listing->slug)) }}" target="_blank" class="w-full bg-primary hover:bg-green-600 text-white font-bold py-3.5 rounded-xl shadow-lg shadow-primary/20 transition-all flex items-center justify-center gap-2 group">
                            <i class="fa-brands fa-whatsapp text-xl"></i>
                            <span>Chat on WhatsApp</span>
                        </a>
                        @endif
                        
                        @if(!empty($seller->phone))
                        <a href="tel:{{ $seller->phone }}" 
                           data-phone="{{ $seller->phone }}"
                           data-revealed="false"
                           onclick="
                               if(this.dataset.revealed === 'false') {
                                   this.innerHTML = '<i class=\'fa-solid fa-phone\'></i> ' + this.dataset.phone;
                                   this.classList.remove('bg-white', 'text-primary', 'border-primary', 'hover:bg-primary/5');
                                   this.classList.add('bg-green-600', 'text-white', 'border-transparent', 'hover:bg-green-700');
                                   this.dataset.revealed = 'true';
                                   return false;
                               }
                           " 
                           class="w-full bg-white border-2 border-primary text-primary hover:bg-primary/5 font-bold py-3.5 rounded-xl transition-all flex items-center justify-center gap-2">
                            <i class="fa-solid fa-phone"></i>
                            <span class="">Show Phone Number</span>
                        </a>
                        @endif

                         @auth('web')
                            @if(auth('web')->id() != $listing->created_by_user_id)
                            <a href="{{ route('web.chat.start', $listing->id) }}" class="w-full bg-gray-50 hover:bg-gray-100 text-gray-700 font-semibold py-3.5 rounded-xl transition-colors flex items-center justify-center gap-2">
                                <i class="fa-regular fa-comment-dots"></i>
                                <span>Chat on Yilick</span>
                            </a>
                            @endif
                         @else
                            <a href="{{ route('login') }}" class="w-full bg-gray-50 hover:bg-gray-100 text-gray-700 font-semibold py-3.5 rounded-xl transition-colors flex items-center justify-center gap-2">
                                <i class="fa-regular fa-comment-dots"></i>
                                <span>Chat on Yilick</span>
                            </a>
                         @endauth
                    </div>

                    <div class="mt-6 pt-6 border-t border-gray-100">
                        <h4 class="text-xs font-bold text-gray-400 uppercase tracking-wider mb-3">Safety Tips</h4>
                        <ul class="space-y-2 text-sm text-gray-500">
                            <li class="flex items-start gap-2">
                                <i class="fa-solid fa-check text-green-500 mt-1"></i>
                                <span>Meet in a safe details public place</span>
                            </li>
                            <li class="flex items-start gap-2">
                                <i class="fa-solid fa-check text-green-500 mt-1"></i>
                                <span>Check the item before paying</span>
                            </li>
                            <li class="flex items-start gap-2">
                                <i class="fa-solid fa-check text-green-500 mt-1"></i>
                                <span>Pay only after collecting the item</span>
                            </li>
                        </ul>
                    </div>
                    
                    <div class="mt-4 text-center">
                        <button onclick="openReportModal()" class="text-xs font-medium text-red-500 hover:text-red-600 flex items-center justify-center gap-1 mx-auto">
                            <i class="fa-regular fa-flag"></i> Report this ad
                        </button>
                        @if($seller->slug && auth('web')->id() != $listing->created_by_user_id)
                        <button onclick="openBlockModal()" class="mt-3 text-xs font-medium text-gray-400 hover:text-gray-600 flex items-center justify-center gap-1 mx-auto transition-colors">
                            <i class="fa-solid fa-ban"></i> Block this Business
                        </button>
                        @endif
                    </div>
                </div>

            </div>
        </div>

        <!-- Related Ads -->
        @if($relatedListings->isNotEmpty())
        <div class="mt-16">
             <div class="flex items-center justify-between mb-8">
                <h2 class="text-1xl font-bold text-gray-900">Similar Listings</h2>
            </div>
            
            <div class="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 md:gap-6">
                 @foreach($relatedListings as $relatedListing)
                    <x-listing-card :listing="$relatedListing" />
                @endforeach
            </div>
        </div>
        @endif
        
    </div>

    <!-- Report Modal -->
    <div id="report-modal" class="fixed inset-0 z-50 hidden transition-opacity duration-300" aria-labelledby="modal-title" role="dialog" aria-modal="true">
        <!-- Backdrop -->
        <div class="fixed inset-0 bg-gray-900/60 backdrop-blur-sm transition-opacity opacity-0" id="report-backdrop" onclick="closeReportModal()"></div>

        <!-- Modal Panel -->
        <div class="fixed inset-0 z-10 w-screen overflow-y-auto">
            <div class="flex min-h-full items-center justify-center p-4 text-center sm:p-0">
                <div class="relative transform overflow-hidden rounded-2xl bg-white text-left shadow-2xl transition-all sm:my-8 sm:w-full sm:max-w-md opacity-0 scale-95" id="report-panel">
                    
                    <!-- Header -->
                    <div class="bg-white px-6 pt-6 pb-4">
                        <div class="flex items-start gap-4">
                            <div class="flex-shrink-0 flex items-center justify-center h-12 w-12 rounded-full bg-red-50 border border-red-100">
                                <i class="fa-solid fa-triangle-exclamation text-red-500 text-xl"></i>
                            </div>
                            <div class="flex-1">
                                <h3 class="text-lg font-bold text-gray-900 leading-6" id="modal-title">Report this Ad</h3>
                                <p class="mt-1 text-sm text-gray-500">Help us keep Yilick safe. What's wrong with this listing?</p>
                            </div>
                            <button onclick="closeReportModal()" class="text-gray-400 hover:text-gray-600 transition-colors">
                                <i class="fa-solid fa-xmark text-xl"></i>
                            </button>
                        </div>
                    </div>

                    <!-- Body -->
                    <div class="px-6 pb-6">
                        <form id="report-form" class="space-y-4">
                            <div>
                                <label for="report-reason" class="block text-sm font-semibold text-gray-700 mb-1.5">Reason</label>
                                <div class="relative">
                                    <select id="report-reason" name="reason" class="block w-full rounded-xl border-gray-200 bg-gray-50/50 text-gray-700 py-3 pl-4 pr-10 focus:border-red-500 focus:bg-white focus:ring-2 focus:ring-red-500/20 transition-all appearance-none text-sm font-medium">
                                        <option value="" disabled selected>Select a reason...</option>
                                        <option value="Fraud / Scam">Fraud / Scam</option>
                                        <option value="Duplicate Ad">Duplicate Ad</option>
                                        <option value="Wrong Category">Wrong Category</option>
                                        <option value="Offensive Content">Offensive Content</option>
                                        <option value="Item Sold">Item Sold</option>
                                        <option value="Other">Other</option>
                                    </select>
                                    <div class="pointer-events-none absolute inset-y-0 right-0 flex items-center px-4 text-gray-500">
                                        <i class="fa-solid fa-chevron-down text-xs"></i>
                                    </div>
                                </div>
                            </div>

                            <div>
                                <label for="report-explanation" class="block text-sm font-semibold text-gray-700 mb-1.5">Additional Details <span class="text-gray-400 font-normal">(Optional)</span></label>
                                <textarea id="report-explanation" name="explanation" rows="4" class="block w-full rounded-xl border-gray-200 bg-gray-50/50 text-gray-700 p-4 focus:border-red-500 focus:bg-white focus:ring-2 focus:ring-red-500/20 transition-all text-sm resize-none placeholder-gray-400" placeholder="Please provide any extra context ensuring specific details..."></textarea>
                            </div>
                        </form>
                    </div>

                    <!-- Footer -->
                    <div class="bg-gray-50 px-6 py-4 flex flex-row-reverse gap-3 border-t border-gray-100">
                        <button type="button" onclick="submitReport()" class="inline-flex w-full sm:w-auto justify-center rounded-xl bg-red-600 px-6 py-2.5 text-sm font-bold text-white shadow-lg shadow-red-600/20 hover:bg-red-700 hover:shadow-red-600/30 transition-all transform active:scale-95">
                            Submit Report
                        </button>
                        <button type="button" onclick="closeReportModal()" class="inline-flex w-full sm:w-auto justify-center rounded-xl bg-white px-6 py-2.5 text-sm font-bold text-gray-700 shadow-sm border border-gray-200 hover:bg-gray-50 hover:text-gray-900 transition-all">
                            Cancel
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Block Business Modal -->
    <div id="block-modal" class="fixed inset-0 z-50 hidden transition-opacity duration-300" aria-labelledby="modal-title" role="dialog" aria-modal="true">
        <!-- Backdrop -->
        <div class="fixed inset-0 bg-gray-900/60 backdrop-blur-sm transition-opacity opacity-0" id="block-backdrop" onclick="closeBlockModal()"></div>

        <!-- Modal Panel -->
        <div class="fixed inset-0 z-10 w-screen overflow-y-auto">
            <div class="flex min-h-full items-center justify-center p-4 text-center sm:p-0">
                <div class="relative transform overflow-hidden rounded-2xl bg-white text-left shadow-2xl transition-all sm:my-8 sm:w-full sm:max-w-md opacity-0 scale-95" id="block-panel">
                    
                    <!-- Header -->
                    <div class="bg-white px-6 pt-6 pb-4">
                        <div class="flex items-start gap-4">
                            <div class="flex-shrink-0 flex items-center justify-center h-12 w-12 rounded-full bg-red-50 border border-red-100">
                                <i class="fa-solid fa-ban text-red-500 text-xl"></i>
                            </div>
                            <div class="flex-1">
                                <h3 class="text-lg font-bold text-gray-900 leading-6">Block Business?</h3>
                                <p class="mt-1 text-sm text-gray-500">Are you sure you want to block <strong>{{ $seller->name }}</strong>? You will no longer see their listings.</p>
                            </div>
                            <button onclick="closeBlockModal()" class="text-gray-400 hover:text-gray-600 transition-colors">
                                <i class="fa-solid fa-xmark text-xl"></i>
                            </button>
                        </div>
                    </div>

                    <!-- Footer -->
                    <div class="bg-gray-50 px-6 py-4 flex flex-row-reverse gap-3 border-t border-gray-100">
                        <button type="button" onclick="submitBlock()" class="inline-flex w-full sm:w-auto justify-center rounded-xl bg-red-600 px-6 py-2.5 text-sm font-bold text-white shadow-lg shadow-red-600/20 hover:bg-red-700 hover:shadow-red-600/30 transition-all transform active:scale-95">
                            Block Business
                        </button>
                        <button type="button" onclick="closeBlockModal()" class="inline-flex w-full sm:w-auto justify-center rounded-xl bg-white px-6 py-2.5 text-sm font-bold text-gray-700 shadow-sm border border-gray-200 hover:bg-gray-50 hover:text-gray-900 transition-all">
                            Cancel
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

    <!-- Lightbox Modal -->
    <div id="lightbox" class="fixed inset-0 z-[100] bg-black/95 hidden transition-opacity duration-300 opacity-0" aria-modal="true" role="dialog">
        <!-- Image Container (Clickable Background) -->
        <div id="lightbox-wrapper" class="absolute inset-0 flex items-center justify-center p-4 cursor-zoom-out">
            <img id="lightbox-image" src="" onerror="this.onerror=null; this.src='/assets/icons/listing_placeholder.svg'" alt="Full Screen View" class="max-w-full max-h-full object-contain scale-95 transition-transform duration-300 select-none cursor-default" onclick="event.stopPropagation()">
        </div>

        <!-- Close Button -->
        <button onclick="AdGallery.closeLightbox()" class="absolute top-4 right-4 z-[120] w-12 h-12 flex items-center justify-center rounded-full bg-black/60 text-white hover:bg-white hover:text-black transition-all backdrop-blur-md shadow-lg border border-white/10 cursor-pointer">
            <i class="fa-solid fa-xmark text-2xl"></i>
        </button>

        <!-- Controls -->
        <!-- Prev -->
        <button onclick="AdGallery.prev()" class="absolute left-4 top-1/2 -translate-y-1/2 z-[120] w-14 h-14 flex items-center justify-center rounded-full bg-black/60 text-white hover:bg-white hover:text-black transition-all backdrop-blur-md shadow-lg border border-white/10 group cursor-pointer">
            <i class="fa-solid fa-chevron-left text-2xl group-hover:-translate-x-0.5 transition-transform"></i>
        </button>
        
        <!-- Next -->
        <button onclick="AdGallery.next()" class="absolute right-4 top-1/2 -translate-y-1/2 z-[120] w-14 h-14 flex items-center justify-center rounded-full bg-black/60 text-white hover:bg-white hover:text-black transition-all backdrop-blur-md shadow-lg border border-white/10 group cursor-pointer">
            <i class="fa-solid fa-chevron-right text-2xl group-hover:translate-x-0.5 transition-transform"></i>
        </button>
        
        <!-- Counter -->
        <div class="absolute bottom-6 left-1/2 -translate-x-1/2 z-[120] px-4 py-2 rounded-full bg-black/60 backdrop-blur-md border border-white/10 text-white/90 font-medium tracking-widest text-sm pointer-events-none">
            <span id="lightbox-counter">1</span> / {{ $listing->media->count() }}
        </div>
    </div>
@endsection

@push('scripts')
<script>
    function openReportModal() {
        const modal = document.getElementById('report-modal');
        const backdrop = document.getElementById('report-backdrop');
        const panel = document.getElementById('report-panel');

        modal.classList.remove('hidden');
        
        // Small delay to allow display:block to apply before transition
        requestAnimationFrame(() => {
            backdrop.classList.remove('opacity-0');
            panel.classList.remove('opacity-0', 'scale-95');
            panel.classList.add('scale-100');
        });
        document.body.style.overflow = 'hidden';
    }

    function closeReportModal() {
        const modal = document.getElementById('report-modal');
        const backdrop = document.getElementById('report-backdrop');
        const panel = document.getElementById('report-panel');

        backdrop.classList.add('opacity-0');
        panel.classList.add('opacity-0', 'scale-95');
        panel.classList.remove('scale-100');

        setTimeout(() => {
             modal.classList.add('hidden');
             document.body.style.overflow = '';
        }, 300); // Match transition duration
    }

    function submitReport() {
        const reason = document.getElementById('report-reason').value;
        const explanation = document.getElementById('report-explanation').value;
        
        if(!reason) {
            alert('Please select a reason.');
            return;
        }

        // Show loading state (optional UI enhancement: disable button)
        const btn = document.querySelector('#report-modal button[onclick="submitReport()"]');
        const originalText = btn.innerText;
        btn.innerText = 'Sending...';
        btn.disabled = true;

        fetch('{{ route('listing.report', $listing->slug) }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            },
            body: JSON.stringify({ reason, explanation })
        })
        .then(response => {
            if (!response.ok) throw new Error('Network response was not ok');
            return response.json();
        })
        .then(data => {
            alert('Thank you! Your report has been submitted.');
            closeReportModal();
            // Reset form
            document.getElementById('report-form').reset();
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Something went wrong. Please try again.');
        })
        .finally(() => {
            btn.innerText = originalText;
            btn.disabled = false;
        });
    }
</script>
<script>
    window.shareAd = function(title, url) {
        if (navigator.share) {
            navigator.share({
                title: title,
                text: 'Check out this ad on Yilick!',
                url: url
            }).catch(console.error);
        } else {
            navigator.clipboard.writeText(url).then(() => {
                alert('Link copied to clipboard!');
            }).catch(console.error);
        }
    };

    window.toggleDescription = function() {
        const content = document.getElementById('description-content');
        const gradient = document.getElementById('description-gradient');
        const toggleText = document.getElementById('description-toggle-text');
        const toggleIcon = document.getElementById('description-toggle-icon');
        
        if (content.classList.contains('max-h-32')) {
            // Expand
            content.classList.remove('max-h-32');
            content.classList.add('max-h-full');
            gradient.classList.add('hidden');
            toggleText.innerText = 'Read less';
            toggleIcon.classList.add('rotate-180');
        } else {
            // Collapse
            content.classList.add('max-h-32');
            content.classList.remove('max-h-full');
            gradient.classList.remove('hidden');
            toggleText.innerText = 'Read more';
            toggleIcon.classList.remove('rotate-180');
            // Scroll back to top of description if needed, or leave as is
             document.getElementById('description-content').scrollIntoView({ behavior: 'smooth', block: 'nearest' });
        }
    };
</script>
<script>
    window.AdGallery = (function() {
        // Private State
        let images = [];
        let currentIndex = 0;
        let isLightboxOpen = false;

        // Private DOM Helper
        const getEl = (id) => document.getElementById(id);
        const placeholder = '/assets/icons/listing_placeholder.svg';

        // Helper to set src with error handling
        const setSrc = (el, src) => {
            if(!el) return;
            // Reset error handler for the new load
            el.onerror = function() {
                this.onerror = null; // Prevent loop
                this.src = placeholder;
            };
            el.src = src;
        };

        // Core UI Update Logic
        function updateUI() {
            if(images.length === 0) return;
            const url = images[currentIndex];

            // 1. Update Main Image
            setSrc(getEl('main-image'), url);
            setSrc(getEl('gallery-bg'), url);
            
            const counter = getEl('current-index');
            if(counter) counter.innerText = currentIndex + 1;

            // 2. Update Thumbnails
            document.querySelectorAll('[id^="thumb-"]').forEach((el, idx) => {
                if(idx === currentIndex) {
                     el.className = "w-16 h-16 flex-shrink-0 rounded-lg border-2 border-primary ring-2 ring-primary/20 overflow-hidden transition-all relative scale-105";
                     el.scrollIntoView({ behavior: 'smooth', block: 'nearest', inline: 'center' });
                } else {
                     el.className = "w-16 h-16 flex-shrink-0 rounded-lg border-2 border-transparent opacity-70 hover:opacity-100 overflow-hidden transition-all relative hover:scale-105";
                }
            });

            // 3. Update Lightbox (if open)
            if(isLightboxOpen) {
                const lbImg = getEl('lightbox-image');
                const lbCount = getEl('lightbox-counter');
                setSrc(lbImg, url);
                if(lbCount) lbCount.innerText = currentIndex + 1;
            }
        }

        return {
            init: function(imageData) {
                images = imageData;
                currentIndex = 0;
                updateUI();
                
                // Keyboard Listeners (Singleton)
                if(!window.adGalleryListenersAttached) {
                    document.addEventListener('keydown', (e) => {
                        if(!isLightboxOpen) return; // Only listen if open
                        if (e.key === 'ArrowLeft') this.prev();
                        if (e.key === 'ArrowRight') this.next();
                        if (e.key === 'Escape') this.closeLightbox();
                    });
                    
                    // Close lightbox on clicking background or wrapper
                    getEl('lightbox')?.addEventListener('click', (e) => {
                         // Check if target is lightbox or the wrapper div
                        if(e.target === getEl('lightbox') || e.target.id === 'lightbox-wrapper') {
                             this.closeLightbox();
                        }
                    });
                    
                    // Touch Support
                    let touchStartX = 0;
                    let touchEndX = 0;
                    const lightboxEl = getEl('lightbox');
                    
                    if(lightboxEl) {
                         lightboxEl.addEventListener('touchstart', e => {
                             touchStartX = e.changedTouches[0].screenX;
                         }, {passive: true});
                         
                         lightboxEl.addEventListener('touchend', e => {
                             touchEndX = e.changedTouches[0].screenX;
                             // Swipe Threshold
                             if (touchEndX < touchStartX - 50) this.next();
                             if (touchEndX > touchStartX + 50) this.prev();
                         }, {passive: true});
                    }

                    window.adGalleryListenersAttached = true;
                }
            },

            next: function() {
                if(images.length <= 1) return;
                currentIndex = (currentIndex + 1) % images.length;
                updateUI();
            },

            prev: function() {
                if(images.length <= 1) return;
                currentIndex = (currentIndex - 1 + images.length) % images.length;
                updateUI();
            },

            setIndex: function(index) {
                currentIndex = index;
                updateUI();
            },

            openLightbox: function() {
                if(images.length === 0) return;
                isLightboxOpen = true;
                
                const box = getEl('lightbox');
                const img = getEl('lightbox-image');
                
                // Sync image before showing
                setSrc(img, images[currentIndex]);
                
                if(box) {
                    box.classList.remove('hidden');
                    // Animation Frame needed for transition
                    requestAnimationFrame(() => {
                        box.classList.remove('opacity-0');
                        if(img) {
                             img.classList.remove('scale-95');
                             img.classList.add('scale-100');
                        }
                    });
                    document.body.style.overflow = 'hidden';
                }
            },

            closeLightbox: function() {
                isLightboxOpen = false;
                const box = getEl('lightbox');
                const img = getEl('lightbox-image');

                if(box) {
                    box.classList.add('opacity-0');
                    if(img) {
                        img.classList.remove('scale-100');
                        img.classList.add('scale-95');
                    }
                    setTimeout(() => {
                        box.classList.add('hidden');
                        document.body.style.overflow = '';
                    }, 300);
                }
            }
        };
    })();

    // Initialize with data from server
    document.addEventListener('DOMContentLoaded', () => {
        if (window.AdGallery) {
            AdGallery.init(@json($listing->media->pluck('url')));
        }
    });

    window.handleAdAction = function(action, id) {
        let confirmMsg = action === 'delete' 
            ? 'Are you sure you want to permanently delete this ad?' 
            : 'Are you sure you want to close this ad?';
            
        if (!confirm(confirmMsg)) return;

        const url = action === 'delete' 
            ? `/listing/${id}` 
            : `/listing/${id}/${action}`;
            
        const method = action === 'delete' ? 'DELETE' : 'POST';

        if (window.showLoading) window.showLoading();

        fetch(url, {
            method: method,
            headers: {
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'Accept': 'application/json',
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                if (action === 'delete') {
                    window.location.href = '{{ route('web.my-ads') }}';
                } else {
                    window.location.reload();
                }
            } else if (window.showToast) {
                window.showToast(data.message || 'Action failed', 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            if (window.showToast) window.showToast('Something went wrong', 'error');
        })
        .finally(() => {
            if (window.hideLoading) window.hideLoading();
        });
    }

    // Block Modal Logic
    function openBlockModal() {
        @auth('web')
            const modal = document.getElementById('block-modal');
            const backdrop = document.getElementById('block-backdrop');
            const panel = document.getElementById('block-panel');

            modal.classList.remove('hidden');
            
            requestAnimationFrame(() => {
                backdrop.classList.remove('opacity-0');
                panel.classList.remove('opacity-0', 'scale-95');
                panel.classList.add('scale-100');
            });
            document.body.style.overflow = 'hidden';
        @else
            window.location.href = "{{ route('login') }}";
        @endauth
    }

    function closeBlockModal() {
        const modal = document.getElementById('block-modal');
        const backdrop = document.getElementById('block-backdrop');
        const panel = document.getElementById('block-panel');

        backdrop.classList.add('opacity-0');
        panel.classList.add('opacity-0', 'scale-95');
        panel.classList.remove('scale-100');

        setTimeout(() => {
             modal.classList.add('hidden');
             document.body.style.overflow = '';
        }, 300);
    }

    function submitBlock() {
        @if($seller->slug)
            const btn = document.querySelector('#block-modal button[onclick="submitBlock()"]');
            const originalText = btn.innerText;
            btn.innerText = 'Blocking...';
            btn.disabled = true;

            fetch('{{ route('business.block', $seller->slug) }}', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                }
            })
            .then(async response => {
                const data = await response.json().catch(() => ({})); // Handle non-JSON responses
                if (!response.ok) {
                    throw new Error(data.message || response.statusText || 'Network response was not ok');
                }
                return data;
            })
            .then(data => {
                alert('Business blocked successfully.');
                window.location.reload(); 
            })
            .catch(error => {
                console.error('Error:', error);
                alert(error.message || 'Something went wrong. Please try again.');
                btn.innerText = originalText;
                btn.disabled = false;
            });
        @endif
    }
</script>
@endpush
