@extends('layouts.app')

@section('title', 'Chat')

@push('styles')
<style>
    /* Fixed layout adjustments */
    body {
        overflow: hidden !important;
    }
    
    footer, 
    nav.fixed.bottom-0,
    nav[class*="bottom-0"],
    #topbars-container { 
        display: none !important;
    }

    main {
        padding-bottom: 0 !important;
        min-height: 0 !important;
    }

    .chat-container-fixed {
        height: calc(100dvh - 64px);
    }

    @media (max-width: 768px) {
        .chat-container-fixed {
            height: calc(100dvh - 56px);
        }
    }

    /* WhatsApp Doodle Background */
    .chat-bg {
        background-color: #e5ddd5;
        background-image: url("https://user-images.githubusercontent.com/15075759/28719144-86dc0f70-73b1-11e7-911d-60d70fcded21.png");
        background-size: 400px;
        background-repeat: repeat;
        background-blend-mode: overlay;
        position: relative;
    }

    /* Date Divider */
    .date-divider {
        display: flex;
        justify-content: center;
        margin: 1.5rem 0;
        position: relative;
    }

    .date-badge {
        background: #e1f3fb;
        color: #5b717b;
        padding: 6px 14px;
        border-radius: 8px;
        font-size: 11px;
        font-weight: 700;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        box-shadow: 0 1px 0.5px rgba(0,0,0,0.08);
    }

    /* WhatsApp Style Bubbles */
    .message-row {
        display: flex;
        flex-direction: column;
        width: 100%;
        margin-bottom: 2px;
    }

    .message-bubble {
        max-width: 85%;
        padding: 6px 8px 8px 10px;
        font-size: 14.2px;
        line-height: 1.4;
        position: relative;
        box-shadow: 0 1px 0.5px rgba(0,0,0,0.13);
        word-wrap: break-word;
    }

    .message-me {
        background-color: #dcf8c6;
        color: #1a1a1a;
        align-self: flex-end;
        border-radius: 8px 0 8px 8px;
    }

    .message-them {
        background-color: #ffffff;
        color: #1a1a1a;
        align-self: flex-start;
        border-radius: 0 8px 8px 8px;
    }

    /* Tails */
    .has-tail-me::after {
        content: "";
        position: absolute;
        top: 0;
        right: -8px;
        width: 0;
        height: 0;
        border-left: 10px solid #dcf8c6;
        border-bottom: 10px solid transparent;
    }

    .has-tail-them::after {
        content: "";
        position: absolute;
        top: 0;
        left: -8px;
        width: 0;
        height: 0;
        border-right: 10px solid #ffffff;
        border-bottom: 10px solid transparent;
    }

    /* Meta Info (Status/Time) */
    .bubble-meta {
        font-size: 10px;
        color: rgba(0, 0, 0, 0.45);
        display: flex;
        align-items: center;
        gap: 2px;
        float: right;
        margin-top: 4px;
        margin-left: 8px;
    }

    .me-meta {
        color: rgba(0, 0, 0, 0.45);
    }

    /* Input Bar Refinement */
    .modern-input-wrapper {
        background: transparent;
        padding: 8px 12px;
        display: flex;
        align-items: flex-end;
        gap: 8px;
    }

    .floating-bar {
        background: white;
        border-radius: 24px;
        flex: 1;
        display: flex;
        align-items: center;
        padding: 5px 14px;
        box-shadow: 0 1px 2px rgba(0,0,0,0.1);
    }

    .send-circle {
        width: 46px;
        height: 46px;
        background: #00cc50;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        box-shadow: 0 1px 2px rgba(0,0,0,0.15);
        flex-shrink: 0;
    }

    /* Override global focus styles for the chat input */
    #message-input:focus {
        outline: none !important;
        border: none !important;
        box-shadow: none !important;
    }

    /* Skeleton Loader */
    @keyframes shimmer {
        0% { background-position: -200% 0; }
        100% { background-position: 200% 0; }
    }
    .skeleton {
        background: linear-gradient(90deg, #f1f5f9 25%, #e2e8f0 50%, #f1f5f9 75%);
        background-size: 200% 100%;
        animation: shimmer 1.5s infinite;
    }
    /* Emoji Picker Styles */
    .emoji-picker:not(.hidden) {
        display: grid;
        grid-template-columns: repeat(6, 1fr);
    }
    .emoji-picker {
        position: absolute;
        bottom: 70px;
        left: 12px;
        background: white;
        border: 1px solid #e2e8f0;
        border-radius: 12px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        padding: 8px;
        gap: 2px;
        z-index: 50;
        width: 250px;
        max-height: 200px;
        overflow-y: auto;
    }
    .emoji-btn {
        font-size: 20px;
        padding: 4px;
        border-radius: 6px;
        transition: background 0.2s;
        display: flex;
        align-items: center;
        justify-content: center;
    }
    .emoji-btn:hover {
        background: #f1f5f9;
    }
    /* Loading Dots Animation */
    .sending-dots {
        display: inline-flex;
        gap: 2px;
        align-items: center;
    }
    .sending-dot {
        width: 4px;
        height: 4px;
        background: #00cc50; /* Changed to brand color for better visibility */
        border-radius: 50%;
        animation: dot-pulse 1.4s infinite;
    }
    .sending-dot:nth-child(2) { animation-delay: 0.2s; }
    .sending-dot:nth-child(3) { animation-delay: 0.4s; }
    
    @keyframes dot-pulse {
        0%, 100% { transform: scale(1); opacity: 0.4; }
        50% { transform: scale(1.3); opacity: 1; }
    }
</style>
@endpush

@section('content')
<div class="chat-container-fixed chat-bg flex flex-col font-sans relative overflow-hidden">
    
    <div class="container mx-auto px-0 md:px-4 max-w-4xl flex flex-col h-full bg-transparent shadow-none md:rounded-none overflow-hidden">
        <!-- Professional Header -->
        <div class="bg-[#f0f2f5] border-b border-gray-200 px-3 py-2 md:px-5 md:py-2.5 flex items-center justify-between z-20">
            <div class="flex items-center gap-2">
                <a href="{{ route('web.chat.inbox') }}" class="p-1 px-2 text-gray-600 hover:bg-gray-200 rounded-full transition-all">
                    <i class="fa-solid fa-arrow-left"></i>
                </a>
                
                <div class="flex items-center gap-2.5 cursor-pointer">
                    <div class="relative">
                        @if(auth()->id() == $conversation->buyer_id)
                            <img src="{{ $conversation->business->logo_url ? $conversation->business->logo_url : asset('assets/icons/favicon.png') }}" onerror="this.src='{{ asset('assets/icons/favicon.png') }}'" class="w-10 h-10 rounded-full object-cover">
                        @else
                            <img src="{{ $conversation->buyer->profile_photo_url ? $conversation->buyer->profile_photo_url : asset('assets/icons/favicon.png') }}" onerror="this.src='{{ asset('assets/icons/favicon.png') }}'" class="w-10 h-10 rounded-full object-cover">
                        @endif
                    </div>
                    <div class="flex flex-col">
                        <span class="font-bold text-gray-800 text-[15px] leading-tight">
                            @if(auth()->id() == $conversation->buyer_id)
                                {{ $conversation->business->name }}
                            @else
                                {{ $conversation->buyer->full_name ?? $conversation->buyer->username }}
                            @endif
                        </span>
                        <div class="flex items-center gap-1">
                            <div class="w-1.5 h-1.5 rounded-full bg-[#00cc50]"></div>
                            <span class="text-[11px] text-gray-500 font-medium">Online</span>
                        </div>
                    </div>
                </div>
            </div>

            <div class="flex items-center gap-1">
                @php
                    $phone = auth()->id() == $conversation->buyer_id ? $conversation->business->primary_phone : $conversation->buyer->phone;
                @endphp
                @if($phone)
                <a href="tel:{{ $phone }}" class="p-2.5 text-gray-600 hover:bg-gray-200 rounded-full transition-all">
                    <i class="fa-solid fa-phone text-sm"></i>
                </a>
                @endif
            </div>
        </div>

        <!-- Sticky Product Info -->
        <div class="bg-white/95 backdrop-blur-sm border-b border-gray-100 px-4 py-2 flex items-center gap-3 z-10">
            <a href="{{ route('listing.show', $conversation->listing->slug) }}" class="flex items-center gap-3 w-full group">
                <div class="w-10 h-10 rounded-lg overflow-hidden border border-gray-100">
                    <img src="{{ $conversation->listing->media->first()->url ?? asset('assets/icons/listing_placeholder.svg') }}" onerror="this.src='{{ asset('assets/icons/listing_placeholder.svg') }}'" class="w-full h-full object-cover">
                </div>
                <div class="flex-1 min-w-0">
                    <p class="text-[12px] font-bold text-gray-900 truncate group-hover:text-primary transition-colors">
                        {{ $conversation->listing->title }}
                    </p>
                    <p class="text-[11px] text-[#008069] font-black">
                        {{ $conversation->listing->price_formatted }}
                    </p>
                </div>
                <div class="bg-primary/10 text-primary text-[10px] font-black px-2 py-1 rounded-md">VIEW ITEM</div>
            </a>
        </div>

        <!-- Messages Flow -->
        <div class="flex-1 overflow-y-auto px-4 py-2 space-y-1" id="messages-container" style="scrollbar-width: thin;">
            <div id="loading" class="flex flex-col gap-2 py-4">
                 @for ($i = 0; $i < 4; $i++)
                 <div class="flex w-full {{ $i % 2 == 0 ? 'justify-start' : 'justify-end' }}">
                     <div class="w-2/3 h-12 skeleton rounded-xl opacity-40"></div>
                 </div>
                 @endfor
            </div>
            <div id="messages-list" class="flex flex-col">
                <!-- Messages populated via JS -->
            </div>
            <div id="scroll-anchor" class="h-4"></div>
        </div>

        <!-- Input Section -->
        <div class="bg-transparent px-2 pb-2 pt-1 relative">
            <!-- Emoji Picker -->
            <div id="emoji-picker" class="emoji-picker hidden animate-in fade-in slide-in-from-bottom-2 duration-200" style="scrollbar-width: none;">
                @php
                    $emojis = ['😀','😃','😄','😁','😆','😅','😂','🤣','😊','😇','🙂','🙃','😉','😌','😍','🥰','😘','😗','😙','😚','😋','😛','😝','😜','🤪','🤨','🧐','🤓','😎','🤩','🥳','😏','😒','😞','😔','😟','😕','🙁','☹️','😣','😖','😫','😩','🥺','😢','😭','😤','😠','😡','🤬','🤯','😳','🥵','🥶','😱','😨','😰','😥','😓','🤗','🤔','🤭','🤫','🤐','😶','😐','😑','😬','🙄','😯','😦','😧','😮','😲','🥱','😴','🤤','😪','😵','🤐','🥴','🤢','🤮','🤧','🤨','🧐','🤓'];
                @endphp
                @foreach($emojis as $emoji)
                    <button type="button" class="emoji-btn" onclick="insertEmoji('{{ $emoji }}')">{{ $emoji }}</button>
                @endforeach
            </div>

            <!-- Image Preview Area -->
            <div id="image-preview" class="hidden mb-2 bg-white/95 backdrop-blur-sm rounded-2xl p-2 mx-2 border border-gray-100 shadow-xl animate-in slide-in-from-bottom-2">
                 <div class="relative inline-block">
                    <img src="" id="preview-img" class="h-24 w-auto rounded-xl border border-gray-100 object-cover shadow-sm">
                    <button type="button" onclick="clearImage()" class="absolute -top-2 -right-2 bg-gray-800 text-white rounded-full w-6 h-6 flex items-center justify-center text-xs shadow-lg">
                        <i class="fa-solid fa-xmark"></i>
                    </button>
                 </div>
            </div>

            <form id="chat-form" class="modern-input-wrapper" onsubmit="sendMessage(event)">
                <div class="floating-bar">
                    <button type="button" onclick="toggleEmojiPicker()" class="p-2 text-[#54656f] hover:text-[#111b21] transition-colors">
                        <i class="fa-regular fa-face-smile text-xl"></i>
                    </button>
                    
                    <input type="file" id="image-upload" accept="image/*" class="hidden" onchange="handleImageSelect(this)">
                    <button type="button" onclick="document.getElementById('image-upload').click()" class="p-2 text-[#54656f] hover:text-[#111b21] transition-colors rotate-45">
                        <i class="fa-solid fa-paperclip text-lg"></i>
                    </button>

                    <div class="flex-1 mx-2">
                        <textarea 
                            id="message-input"
                            rows="1" 
                            placeholder="Type a message" 
                            class="w-full bg-transparent border-none focus:ring-0 p-1 text-[15px] text-[#111b21] placeholder-[#54656f] font-normal resize-none max-h-32 mb-0.5"
                            oninput="autoResize(this)"
                            onkeydown="handleEnter(event)"
                            onfocus="hideEmojiPicker()"
                        ></textarea>
                    </div>
                </div>

                <button type="submit" id="send-btn" class="send-circle hover:scale-105 active:scale-95 transition-all disabled:opacity-50">
                    <i class="fa-solid fa-paper-plane text-base"></i>
                </button>
            </form>
        </div>
    </div>
</div>

@endsection

@push('scripts')
<script>
    const conversationId = {{ $conversation->id }};
    const currentUserId = {{ auth()->id() }};
    let pollingInterval = null;
    let isSending = false;
    let selectedImage = null;
    let lastRenderedDate = null;
    let lastRenderedSender = null;

    document.addEventListener('DOMContentLoaded', () => {
        loadMessages();
        startPolling();
        autoResize(document.getElementById('message-input'));
        
        // Close emoji picker when clicking outside
        document.addEventListener('click', (e) => {
            const picker = document.getElementById('emoji-picker');
            const smileBtn = document.querySelector('button[onclick="toggleEmojiPicker()"]');
            if (!picker.contains(e.target) && !smileBtn.contains(e.target)) {
                hideEmojiPicker();
            }
        });
    });

    function toggleEmojiPicker() {
        document.getElementById('emoji-picker').classList.toggle('hidden');
    }

    function hideEmojiPicker() {
        document.getElementById('emoji-picker').classList.add('hidden');
    }

    function insertEmoji(emoji) {
        const input = document.getElementById('message-input');
        const start = input.selectionStart;
        const end = input.selectionEnd;
        const text = input.value;
        input.value = text.substring(0, start) + emoji + text.substring(end);
        input.focus();
        input.selectionStart = input.selectionEnd = start + emoji.length;
        autoResize(input);
    }

    function handleEnter(e) {
        if(e.key === 'Enter' && !e.shiftKey) {
            e.preventDefault();
            sendMessage(e);
        }
    }

    function autoResize(textarea) {
        textarea.style.height = 'auto';
        textarea.style.height = textarea.scrollHeight + 'px';
    }

    function loadMessages() {
        axios.get(`{{ route('web.chat.messages', $conversation->id) }}`)
            .then(response => {
                const messages = response.data.messages.reverse();
                renderMessages(messages, true);
                document.getElementById('loading').classList.add('hidden');
                
                // Ensure DOM is updated and layout stable before scrolling
                setTimeout(() => {
                    scrollToBottom(false);
                }, 100);
            })
            .catch(console.error);
    }

    function getFormattedDate(dateStr) {
        if (!dateStr) return '';
        const d = new Date(dateStr);
        if (isNaN(d.getTime())) return '';
        
        const now = new Date();
        const yesterday = new Date();
        yesterday.setDate(now.getDate() - 1);

        if (d.toDateString() === now.toDateString()) return 'TODAY';
        if (d.toDateString() === yesterday.toDateString()) return 'YESTERDAY';
        
        return d.toLocaleDateString('en-US', { day: 'numeric', month: 'long', year: 'numeric' });
    }

    function renderMessages(messages, clear = false) {
        const list = document.getElementById('messages-list');
        if (clear) {
            list.innerHTML = '';
            lastRenderedDate = null;
            lastRenderedSender = null;
        }

        messages.forEach(msg => {
            if(document.getElementById(`msg-${msg.id}`)) return;

            // Date divider logic
            const msgDate = getFormattedDate(msg.created_at_raw || msg.created_at);
            if (msgDate !== lastRenderedDate) {
                const divider = document.createElement('div');
                divider.className = 'date-divider';
                divider.innerHTML = `<span class="date-badge">${msgDate}</span>`;
                list.appendChild(divider);
                lastRenderedDate = msgDate;
                lastRenderedSender = null; // Reset sender for tails on new date
            }

            const isTail = msg.sender_id !== lastRenderedSender;
            lastRenderedSender = msg.sender_id;

            const row = document.createElement('div');
            row.id = `msg-${msg.id}`;
            row.className = `message-row animate-in fade-in duration-300 ${isTail ? 'mt-3' : 'mt-0.5'} ${msg.is_sending ? 'opacity-60' : ''}`;
            
            let content = '';
            const bubbleClass = msg.is_me ? 'message-me' : 'message-them';
            const tailClass = isTail ? (msg.is_me ? 'has-tail-me' : 'has-tail-them') : '';
            const time = msg.time_formatted || msg.created_at;

            const statusIcon = msg.is_sending 
                ? '<span class="sending-dots"><span class="sending-dot"></span><span class="sending-dot"></span><span class="sending-dot"></span></span>' 
                : '<i class="fa-solid fa-check-double text-[11px] text-[#53bdeb]"></i>';

            if (msg.type === 'text') {
                content = `
                    <div class="message-bubble ${bubbleClass} ${tailClass}">
                        <span>${escapeHtml(msg.message)}</span>
                        <div class="bubble-meta ${msg.is_me ? 'me-meta' : ''}">
                            <span class="text-[10px]">${time}</span>
                            ${msg.is_me ? statusIcon : ''}
                        </div>
                    </div>
                `;
            } else if (msg.type === 'image') {
                content = `
                    <div class="message-bubble ${bubbleClass} ${tailClass} p-1">
                        <a href="${msg.attachment}" target="_blank" class="block overflow-hidden rounded-lg">
                            <img src="${msg.attachment}" onerror="this.src='{{ asset('assets/icons/listing_placeholder.svg') }}'" class="w-full max-w-[280px] h-auto object-cover max-h-[400px]">
                        </a>
                        <div class="bubble-meta px-1.5 pb-0.5">
                             <span class="text-[10px]">${time}</span>
                             ${msg.is_me ? statusIcon : ''}
                        </div>
                    </div>
                `;
            }

            row.innerHTML = content;
            list.appendChild(row);
        });
    }

    function sendMessage(e) {
        e.preventDefault();
        if (isSending) return;

        const input = document.getElementById('message-input');
        const text = input.value.trim();

        if (!text && !selectedImage) return;

        isSending = true;
        document.getElementById('send-btn').disabled = true;

        // Optimistic UI: Add temporary message
        const tempId = 'temp-' + Date.now();
        const tempMsg = {
            id: tempId,
            message: text,
            attachment: selectedImage ? document.getElementById('preview-img').src : null,
            type: selectedImage ? 'image' : 'text',
            is_me: true,
            sender_id: currentUserId,
            created_at: new Date().toLocaleTimeString([], { hour: 'numeric', minute: '2-digit' }),
            created_at_raw: new Date().toISOString(),
            is_sending: true
        };
        renderMessages([tempMsg]);
        scrollToBottom(true);

        const formData = new FormData();
        formData.append('conversation_id', conversationId);
        formData.append('type', selectedImage ? 'image' : 'text');
        
        if (text) formData.append('message', text);
        if (selectedImage) formData.append('attachment', selectedImage);

        input.value = '';
        input.style.height = 'auto';
        clearImage();

         axios.post('{{ route('web.chat.send') }}', formData)
            .then(response => {
                // Remove temp message and add real one
                const tempEl = document.getElementById(`msg-${tempId}`);
                if(tempEl) tempEl.remove();
                
                // Reset last rendered sender to ensure tail logic re-runs correctly if needed
                lastRenderedSender = null; 
                loadMessages(); // Re-load to ensure correct ordering and tails
            })
            .catch(error => {
                const tempEl = document.getElementById(`msg-${tempId}`);
                if(tempEl) tempEl.classList.add('text-red-500');
                alert('Failed to send');
                if (text) input.value = text;
            })
            .finally(() => {
                isSending = false;
                document.getElementById('send-btn').disabled = false;
            });
    }

    function scrollToBottom(smooth = true) {
        const container = document.getElementById('messages-container');
        container.scrollTo({
            top: container.scrollHeight,
            behavior: smooth ? 'smooth' : 'auto'
        });
    }

    function startPolling() {
        if(pollingInterval) clearInterval(pollingInterval);
        pollingInterval = setInterval(() => {
            if(!isSending) {
                axios.get(`{{ route('web.chat.messages', $conversation->id) }}`)
                .then(response => {
                    const messages = response.data.messages.reverse();
                    const oldCount = document.getElementById('messages-list').querySelectorAll('.message-row').length;
                    renderMessages(messages);
                    if(document.getElementById('messages-list').querySelectorAll('.message-row').length > oldCount) {
                        scrollToBottom(true);
                    }
                });
            }
        }, 5000);
    }

    function handleImageSelect(input) {
        if (input.files && input.files[0]) {
            selectedImage = input.files[0];
            const reader = new FileReader();
            reader.onload = e => {
                document.getElementById('preview-img').src = e.target.result;
                document.getElementById('image-preview').classList.remove('hidden');
                document.getElementById('image-preview').classList.add('flex');
            }
            reader.readAsDataURL(input.files[0]);
        }
    }

    function clearImage() {
        selectedImage = null;
        const preview = document.getElementById('image-preview');
        preview.classList.add('hidden');
        preview.classList.remove('flex');
        document.getElementById('image-upload').value = '';
    }

    function escapeHtml(text) {
        if (!text) return text;
        return text.replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;").replace(/"/g, "&quot;").replace(/'/g, "&#039;");
    }
</script>
@endpush
