<?php

use App\Http\Controllers\Api\V1\AdminController;
use App\Http\Controllers\Api\V1\AdminBusinessFeedbackController;
use App\Http\Controllers\Api\V1\AuthController;
use App\Http\Controllers\Api\V1\BusinessController;
use App\Http\Controllers\Api\V1\BusinessFeedbackController;
use App\Http\Controllers\Api\V1\BusinessFeedbackManagementController;
use App\Http\Controllers\Api\V1\BusinessListingController;
use App\Http\Controllers\Api\V1\BusinessPublicController;
use App\Http\Controllers\Api\V1\ListingPublicController;
use App\Http\Controllers\Api\V1\ReverseImageSearchController;
use App\Http\Controllers\Api\V1\AIListingController;
use App\Http\Controllers\Api\V1\ListingViewsController;
use App\Http\Controllers\Api\V1\ListingAnalyticsController;
use App\Http\Controllers\Api\V1\SearchAnalyticsController;
use App\Http\Controllers\Api\V1\FavoritesController;
use App\Http\Controllers\Api\V1\PremiumPackageController;
use App\Http\Controllers\Api\V1\SubscriptionController;
use App\Http\Controllers\Api\V1\RelworxWebhookController;
use App\Http\Controllers\Api\V1\ListingPromotionController;
use App\Http\Controllers\Api\V1\UserController;
use App\Http\Controllers\Api\V1\ChatController;
use App\Http\Controllers\Api\V1\NotificationController;
use App\Http\Controllers\Api\V1\AdController;
use Illuminate\Support\Facades\Route;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| API-only authentication and user management system for classified platform.
| All routes are versioned under /api/v1
|
*/

Route::prefix('v1')->group(function () {

    /*
    |--------------------------------------------------------------------------
    | Public Authentication Routes (Rate Limited)
    |--------------------------------------------------------------------------
    */
    Route::middleware(['throttle:auth'])->group(function () {
        // Register a new user account and return access + refresh tokens.
        Route::post('/auth/register', [AuthController::class, 'register'])->name('auth.register');

        // Login with email/phone + password and return access + refresh tokens.
        Route::post('/auth/login', [AuthController::class, 'login'])->name('auth.login');

        // Request a password reset email (Laravel broker).
        Route::post('/auth/forgot-password', [AuthController::class, 'forgotPassword'])->name('auth.forgot-password');

        // Reset password using token + email + new password.
        Route::post('/auth/reset-password', [AuthController::class, 'resetPassword'])->name('auth.reset-password');

        // Refresh access token using a valid refresh token.
        Route::post('/auth/refresh', [AuthController::class, 'refresh'])->name('auth.refresh');
    });

    /*
    |--------------------------------------------------------------------------
    | Email Verification Routes (Public with signed URL)
    |--------------------------------------------------------------------------
    */
    // Verify email using a signed link (returns an HTML success/failure page).
    Route::get('/auth/verify-email/{id}/{hash}', [AuthController::class, 'verifyEmail'])
        ->name('verification.verify');

    /*
    |--------------------------------------------------------------------------
    | OTP Routes (Rate Limited)
    |--------------------------------------------------------------------------
    */
    Route::middleware(['throttle:otp'])->group(function () {
        // Generate an OTP and send it via email or SMS depending on type.
        Route::post('/auth/send-otp', [AuthController::class, 'sendOtp'])->name('auth.send-otp');

        // Verify an OTP code for the given identifier (email/phone).
        Route::post('/auth/verify-otp', [AuthController::class, 'verifyOtp'])->name('auth.verify-otp');
    });

    // Public: list approved feedback for a business by its slug.
    Route::get('/businesses/{slug}/feedback', [BusinessPublicController::class, 'feedbackBySlug'])
        ->where('slug', '^(?!\d+$)(?!following$)[A-Za-z0-9\-]+$')
        ->name('businesses.feedback.by-slug');

    // Public: browse listings (search + filter + promoted/bumped ordering).
    Route::get('/listings', [ListingPublicController::class, 'index'])
        ->middleware([\App\Http\Middleware\OptionalAuth::class])
        ->name('listings.index');

    // Public: reverse image search - find similar listings by uploading a photo.
    Route::post('/listings/search-by-image', [ReverseImageSearchController::class, 'search'])->name('listings.search-by-image');

    // Analytics tracking endpoints (public - called from frontend)
    // Public endpoints: Track when a listing is displayed
    Route::post('/analytics/impressions', [ListingAnalyticsController::class, 'trackImpression'])
        ->name('analytics.track-impression');
        // Track when a listing is clicked
    Route::post('/analytics/clicks', [ListingAnalyticsController::class, 'trackClick'])
        ->name('analytics.track-click');
        // Track other user interactions with a listing (whatsapp, phone, share)
    Route::post('/analytics/interactions', [ListingAnalyticsController::class, 'trackInteraction'])
        ->name('analytics.track-interaction');
       

    // Platform analytics (public for advertisers)
    Route::get('/analytics/platform', [ListingAnalyticsController::class, 'getPlatformAnalytics'])
        ->name('analytics.platform');
    // Search analytics (public) 
    // Get popular/trending search queries
    Route::get('/search/popular', [SearchAnalyticsController::class, 'popularSearches'])
        ->name('search.popular');

    // Ad tracking and serving
    Route::get('/ads/random', [AdController::class, 'random'])->name('ads.random');
    Route::post('/ads/{ad}/track', [AdController::class, 'track'])->name('ads.track');

    // Premium packages (public)
    // List all available premium packages
    Route::get('/packages', [PremiumPackageController::class, 'index'])
        ->name('packages.index');
    // View a specific package by slug
    Route::get('/packages/{slug}', [PremiumPackageController::class, 'show'])
        ->name('packages.show');

    // Public: view listing by slug.
    Route::get('/listings/{slug}', [ListingPublicController::class, 'showBySlug'])
        ->where('slug', '^(?!\d+$)[A-Za-z0-9\-]+$')
        ->middleware([\App\Http\Middleware\OptionalAuth::class])
        ->name('listings.show-by-slug');

    Route::get('/businesses/{slug}', [BusinessPublicController::class, 'showBySlug'])
        ->where('slug', '^(?!\d+$)(?!following$)[A-Za-z0-9\-]+$')
        ->name('businesses.show-by-slug');

    // Content Moderation (Public/Optional Auth)
    Route::post('/listings/{listing}/report', [\App\Http\Controllers\Api\V1\UserContentController::class, 'reportListing'])
        ->middleware([\App\Http\Middleware\OptionalAuth::class])
        ->name('listings.report');

    /*
    |--------------------------------------------------------------------------
    | Authenticated Routes
    |--------------------------------------------------------------------------
    */
    Route::middleware(['jwt.auth'])->group(function () {

        // AI-powered listing content generation - generate title, description, and tags.
        Route::post('/listings/generate-content', [AIListingController::class, 'generateContent'])
            ->name('listings.generate-content');

        // AI Smart Pricing
        Route::get('/listings/recommended-price', [AIListingController::class, 'recommendPrice'])
            ->name('listings.recommended-price');

        // Search analytics (authenticated)
        // Get user's search history
        Route::get('/search/history', [SearchAnalyticsController::class, 'userHistory'])
            ->name('search.history');
        
        // Get search analytics (admin/business owners)
        Route::get('/search/analytics', [SearchAnalyticsController::class, 'analytics'])
            ->name('search.analytics');
        
        // Get searches that returned zero results (for improving search)
        Route::get('/search/zero-results', [SearchAnalyticsController::class, 'zeroResultSearches'])
            ->name('search.zero-results');

        // Auth management
        Route::prefix('auth')->group(function () {
            // Get the currently authenticated user.
            Route::get('/me', [AuthController::class, 'me'])->name('auth.me');

            // List active logged-in devices (login sessions) for the current user.
            Route::get('/sessions', [AuthController::class, 'sessions'])->name('auth.sessions');

            // Logout (revoke) a specific device/session by session ID.
            Route::delete('/sessions/{sessionId}', [AuthController::class, 'logoutSession'])->name('auth.logout-session');

            // Logout current device (invalidate current JWT + revoke refresh token).
            Route::post('/logout', [AuthController::class, 'logout'])->name('auth.logout');

            // Logout from all devices (revoke all refresh tokens for user).
            Route::post('/logout-all', [AuthController::class, 'logoutAll'])->name('auth.logout-all');

            // Re-send email verification message for the authenticated user.
            Route::post('/resend-verification', [AuthController::class, 'resendVerification'])
                ->middleware(['throttle:verification'])
                ->name('auth.resend-verification');
        });

        // User profile management
        Route::prefix('user')->group(function () {
            // Fetch profile details for current user.
            Route::get('/profile', [UserController::class, 'profile'])->name('user.profile');

            // Fetch listings for current user across all businesses.
            Route::get('/listings', [\App\Http\Controllers\Api\V1\UserListingController::class, 'index'])->name('user.listings.index');

            // Update profile details (PUT).
            Route::put('/profile', [UserController::class, 'updateProfile'])->name('user.update-profile');

            // Update profile details (POST alternative for clients that can't PUT).
            Route::post('/profile', [UserController::class, 'updateProfile'])->name('user.update-profile-post');

            // Change password for current user.
            Route::put('/change-password', [UserController::class, 'changePassword'])->name('user.change-password');

            // Remove profile photo for current user.
            Route::delete('/profile-photo', [UserController::class, 'removeProfilePhoto'])->name('user.remove-photo');

            // Delete (soft-delete) current user account.
            Route::delete('/account', [UserController::class, 'deleteAccount'])->name('user.delete-account');
        });

        // Favorites management
        Route::prefix('favorites')->group(function () {
            // Business favorites
            //Fetch Favorite Businesses
            Route::get('/businesses', [FavoritesController::class, 'getFavoriteBusinesses'])
                ->name('favorites.businesses.index');
            // Add Business to Favorites
            Route::post('/businesses', [FavoritesController::class, 'addBusinessToFavorites'])
                ->name('favorites.businesses.add');
            // Remove Business from Favorites
            Route::delete('/businesses/{businessId}', [FavoritesController::class, 'removeBusinessFromFavorites'])
                ->whereNumber('businessId')
                ->name('favorites.businesses.remove');

            // Listing favorites
            // Fetch Favorite Listings
            Route::get('/listings', [FavoritesController::class, 'getFavoriteListings'])
                ->name('favorites.listings.index');
            // Add Listing to Favorites
            Route::post('/listings', [FavoritesController::class, 'addListingToFavorites'])
                ->name('favorites.listings.add');
            // Remove Listing from Favorites
            Route::delete('/listings/{listingId}', [FavoritesController::class, 'removeListingFromFavorites'])
                ->whereNumber('listingId')
                ->name('favorites.listings.remove');

            // Saved searches
            // Fetch Favorite Searches
            Route::get('/searches', [FavoritesController::class, 'getFavoriteSearches'])
                ->name('favorites.searches.index');
            // Save Favorite Search
            Route::post('/searches', [FavoritesController::class, 'saveFavoriteSearch'])
                ->name('favorites.searches.save');
            // Update Favorite Search
            Route::put('/searches/{id}', [FavoritesController::class, 'updateFavoriteSearch'])
                ->whereNumber('id')
                ->name('favorites.searches.update');
            //Delete Search from favorites
            Route::delete('/searches/{id}', [FavoritesController::class, 'deleteFavoriteSearch'])
                ->whereNumber('id')
                ->name('favorites.searches.delete');

            // Check if item is favorited
            Route::post('/check', [FavoritesController::class, 'checkIfFavorited'])
                ->name('favorites.check');
        });

        // Content Moderation (Authenticated)
        Route::post('/businesses/{business:slug}/block', [\App\Http\Controllers\Api\V1\UserContentController::class, 'blockBusiness'])
            ->name('businesses.block');

        // Business management (businesses act like “pages” for posting adverts/listings).
        Route::prefix('businesses')->group(function () {
            // List businesses the current user follows.
            Route::get('/following', [BusinessController::class, 'myFollowing'])->name('businesses.following');

            // List businesses the current user owns or manages.
            Route::get('/', [BusinessController::class, 'index'])->name('businesses.index');

        // Create a new business (enforces BUSINESS_MAX_PER_USER) and promote user to seller.
        Route::post('/', [BusinessController::class, 'store'])->name('businesses.store');

        // Get a single business by ID (only owner/members).
        Route::get('/{business}', [BusinessController::class, 'show'])
            ->whereNumber('business')
            ->name('businesses.show');

        // Update a business profile (owner/manager).
        Route::put('/{business}', [BusinessController::class, 'update'])->name('businesses.update');

        // Submit feedback to a business (authenticated user).
        Route::post('/{business}/feedback', [BusinessFeedbackController::class, 'store'])
            ->whereNumber('business')
            ->name('businesses.feedback.store');

        // Update a submitted feedback (author only).
        Route::put('/{business}/feedback/{feedback}', [BusinessFeedbackController::class, 'update'])
            ->whereNumber('business')
            ->whereNumber('feedback')
            ->name('businesses.feedback.update');

            // Update a submitted feedback (author only).
            Route::put('/{business}/feedback/{feedback}', [BusinessFeedbackController::class, 'update'])
                ->whereNumber('business')
                ->whereNumber('feedback')
                ->name('businesses.feedback.update');

            // Delete a submitted feedback (author only).
            Route::delete('/{business}/feedback/{feedback}', [BusinessFeedbackController::class, 'destroy'])
                ->whereNumber('business')
                ->whereNumber('feedback')
                ->name('businesses.feedback.destroy');

            // List feedback for a business (owner/manager).
            Route::get('/{business}/feedback', [BusinessFeedbackManagementController::class, 'index'])
                ->whereNumber('business')
                ->name('businesses.feedback.index');

            // Respond to a feedback item (owner/manager).
            Route::post('/{business}/feedback/{feedback}/responses', [BusinessFeedbackManagementController::class, 'respond'])
                ->whereNumber('business')
                ->whereNumber('feedback')
                ->name('businesses.feedback.respond');

            // Listings management (owner/manager/listings_manager).
            // List listings for a business (with filters for business dashboard).
            Route::get('/{business}/listings', [BusinessListingController::class, 'index'])
                ->whereNumber('business')
                ->name('businesses.listings.index');

            // Pre-upload listing media before creating the listing.
            Route::post('/{business}/listings/media/upload', [BusinessListingController::class, 'uploadMedia'])
                ->whereNumber('business')
                ->name('businesses.listings.media.upload');

            // Create a listing for a business (supports media_ids or direct media upload).
            Route::post('/{business}/listings', [BusinessListingController::class, 'store'])
                ->whereNumber('business')
                ->name('businesses.listings.store');

            // Get a single listing by ID for business management.
            Route::get('/{business}/listings/{listing}', [BusinessListingController::class, 'show'])
                ->whereNumber('business')
                ->whereNumber('listing')
                ->name('businesses.listings.show');

            // Update a listing (supports replacing media + updating tags JSON/custom fields).
            Route::put('/{business}/listings/{listing}', [BusinessListingController::class, 'update'])
                ->whereNumber('business')
                ->whereNumber('listing')
                ->name('businesses.listings.update');

            // Delete (soft-delete) a listing for a business.
            Route::delete('/{business}/listings/{listing}', [BusinessListingController::class, 'destroy'])
                ->whereNumber('business')
                ->whereNumber('listing')
                ->name('businesses.listings.destroy');

            // Get view statistics for a listing.
            Route::get('/{business}/listings/{listing}/views/stats', [ListingViewsController::class, 'stats'])
                ->whereNumber('business')
                ->whereNumber('listing')
                ->name('businesses.listings.views.stats');

            // Get list of viewers for a listing.
            Route::get('/{business}/listings/{listing}/views/viewers', [ListingViewsController::class, 'viewers'])
                ->whereNumber('business')
                ->whereNumber('listing')
                ->name('businesses.listings.views.viewers');

            // Get comprehensive analytics for a listing.
            Route::get('/{business}/listings/{listing}/analytics', [ListingAnalyticsController::class, 'getListingAnalytics'])
                ->whereNumber('business')
                ->whereNumber('listing')
                ->name('businesses.listings.analytics');

            // Bump (push-to-top) a published listing.
            Route::post('/{business}/listings/{listing}/bump', [BusinessListingController::class, 'bump'])
                ->whereNumber('business')
                ->whereNumber('listing')
                ->name('businesses.listings.bump');

            // Inventory Management
            Route::post('/{business}/listings/{listing}/inventory/add', [\App\Http\Controllers\Api\V1\InventoryController::class, 'addStock'])
                ->whereNumber('business')
                ->whereNumber('listing')
                ->name('businesses.listings.inventory.add');

            Route::post('/{business}/listings/{listing}/inventory/transfer', [\App\Http\Controllers\Api\V1\InventoryController::class, 'transferStock'])
                ->whereNumber('business')
                ->whereNumber('listing')
                ->name('businesses.listings.inventory.transfer');

            Route::post('/{business}/listings/{listing}/inventory/remove', [\App\Http\Controllers\Api\V1\InventoryController::class, 'removeStock'])
                ->whereNumber('business')
                ->whereNumber('listing')
                ->name('businesses.listings.inventory.remove');

            Route::get('/{business}/listings/{listing}/inventory/logs', [\App\Http\Controllers\Api\V1\InventoryController::class, 'getLogs'])
                ->whereNumber('business')
                ->whereNumber('listing')
                ->name('businesses.listings.inventory.logs');

            // Delete (soft-delete) a business (owner only).
            Route::delete('/{business}', [BusinessController::class, 'destroy'])->name('businesses.destroy');

            // List business members and their roles.
            Route::get('/{business}/members', [BusinessController::class, 'members'])->name('businesses.members');
            
            // List pending business invitations.
            Route::get('/{business}/invitations', [BusinessController::class, 'invitations'])->name('businesses.invitations');

            // List followers of a business.
            Route::get('/{business}/followers', [BusinessController::class, 'followers'])
                ->whereNumber('business')
                ->name('businesses.followers');

            // Invite a user by email to manage this business.
            Route::post('/{business}/invites', [BusinessController::class, 'invite'])->name('businesses.invite');

            // Update a member's role (owner/manager).
            Route::put('/{business}/members/{userId}/role', [BusinessController::class, 'updateMemberRole'])->name('businesses.members.update-role');

            // Remove a member from the business (owner/manager).
            Route::delete('/{business}/members/{userId}', [BusinessController::class, 'removeMember'])->name('businesses.members.remove');

            // Transfer business ownership to another user (by email).
            Route::post('/{business}/transfer-ownership', [BusinessController::class, 'transferOwnership'])->name('businesses.transfer-ownership');

            // Request verification for a business (sets verification_status to pending).
            Route::post('/{business}/verification/request', [BusinessController::class, 'requestVerification'])->name('businesses.verification.request');

            // Follow a business.
            Route::post('/{business}/follow', [BusinessController::class, 'follow'])->name('businesses.follow');

            // Unfollow a business.
            Route::delete('/{business}/follow', [BusinessController::class, 'unfollow'])->name('businesses.unfollow');

            // Send an OTP to verify a business contact number (owner/manager).
            Route::post('/{business}/contact/send-otp', [BusinessController::class, 'sendContactOtp'])
                ->middleware(['throttle:otp'])
                ->name('businesses.contact.send-otp');

            // Verify an OTP for a business contact number (owner/manager).
            Route::post('/{business}/contact/verify-otp', [BusinessController::class, 'verifyContactOtp'])
                ->middleware(['throttle:otp'])
                ->name('businesses.contact.verify-otp');
        });

        // Accept a business invitation using the invite token (must match authenticated user's email).
        Route::post('/business-invites/{token}/accept', [BusinessController::class, 'acceptInvite'])->name('business-invites.accept');

        // Decline a business invitation using the invite token (must match authenticated user's email).
        Route::post('/business-invites/{token}/decline', [BusinessController::class, 'declineInvite'])->name('business-invites.decline');

        /*
        |--------------------------------------------------------------------------
        | Admin Routes (Role Protected)
        |--------------------------------------------------------------------------
        */
        Route::prefix('admin')->middleware(['role:admin'])->group(function () {
            // User management
            // List users (admin only).
            Route::get('/users', [AdminController::class, 'listUsers'])->name('admin.users.list');

            // Aggregated user statistics (admin only).
            Route::get('/users/statistics', [AdminController::class, 'statistics'])->name('admin.users.statistics');

            // Get a single user by ID (admin only).
            Route::get('/users/{id}', [AdminController::class, 'showUser'])->name('admin.users.show');

            // Update a user's status (active/suspended) (admin only).
            Route::put('/users/{id}/status', [AdminController::class, 'updateUserStatus'])->name('admin.users.update-status');

            // Update a user's role (admin only).
            Route::put('/users/{id}/role', [AdminController::class, 'updateUserRole'])->name('admin.users.update-role');

            // Restore a previously deleted user (admin only).
            Route::post('/users/{id}/restore', [AdminController::class, 'restoreUser'])->name('admin.users.restore');

            // Permanently delete a user (admin only).
            Route::delete('/users/{id}/force', [AdminController::class, 'forceDeleteUser'])->name('admin.users.force-delete');

            // Business feedback moderation (admin only).
            // List pending business feedback for moderation.
            Route::get('/business-feedback/pending', [AdminBusinessFeedbackController::class, 'pending'])
                ->name('admin.business-feedback.pending');

            // Approve a business feedback item.
            Route::post('/business-feedback/{feedback}/approve', [AdminBusinessFeedbackController::class, 'approve'])
                ->whereNumber('feedback')
                ->name('admin.business-feedback.approve');

            // Reject a business feedback item.
            Route::post('/business-feedback/{feedback}/reject', [AdminBusinessFeedbackController::class, 'reject'])
                ->whereNumber('feedback')
                ->name('admin.business-feedback.reject');
        });

        // Chat & Messaging
        Route::prefix('chat')->group(function () {
            // Get inbox
            Route::get('/', [ChatController::class, 'index'])->name('chat.index');
            
            // Get conversation history
            Route::get('/{conversation}', [ChatController::class, 'show'])
                ->whereNumber('conversation')
                ->name('chat.show');
            
            // Send a message
            Route::post('/', [ChatController::class, 'store'])->name('chat.store');
            
            // Mark as read
            Route::post('/{conversation}/read', [ChatController::class, 'markRead'])
                ->whereNumber('conversation')
                ->name('chat.mark-read');
        });

        // Notifications
        Route::prefix('notifications')->group(function () {
            // Update FCM token for push notifications.
            Route::post('/token', [NotificationController::class, 'updateToken'])->name('notifications.update-token');
            
            // Notification History & Management
            Route::get('/', [NotificationController::class, 'index'])->name('notifications.index');
            Route::post('/{id}/read', [NotificationController::class, 'markAsRead'])->name('notifications.mark-read');
            Route::post('/read-all', [NotificationController::class, 'markAllAsRead'])->name('notifications.mark-all-read');
        });

        // Admin Notification Routes
        Route::prefix('admin/notifications')->group(function () {
             Route::post('/send', [\App\Http\Controllers\Api\V1\AdminNotificationController::class, 'send'])->name('admin.notifications.send');
        });

        // POS System Routes
        Route::prefix('pos')->group(function () {
            // Pull updates (Sync items to device)
            Route::get('/{business}/sync', [\App\Http\Controllers\Api\V1\PosController::class, 'pull'])
                ->whereNumber('business')
                ->name('pos.sync.pull');
                
            // Push updates (Sync sales to server)
            Route::post('/{business}/sync', [\App\Http\Controllers\Api\V1\PosController::class, 'push'])
                ->whereNumber('business')
                ->name('pos.sync.push');

            // Bulk SKU Update
            Route::post('/{business}/listings/bulk-sku-update', [\App\Http\Controllers\Api\V1\PosController::class, 'bulkUpdateSkus'])
                ->whereNumber('business')
                ->name('pos.listings.bulk-sku-update');

            // Customer Management (Sync & CRUD)
            Route::get('/{business}/customers', [\App\Http\Controllers\Api\V1\PosCustomerController::class, 'index'])
                ->whereNumber('business')
                ->name('pos.customers.index');
                
            Route::post('/{business}/customers', [\App\Http\Controllers\Api\V1\PosCustomerController::class, 'store'])
                ->whereNumber('business')
                ->name('pos.customers.store');

            Route::post('/{business}/customers/{customer}/payments', [\App\Http\Controllers\Api\V1\PosCustomerController::class, 'recordPayment'])
                ->whereNumber('business')
                ->whereNumber('customer')
                ->name('pos.customers.payments');
                
            Route::post('/{business}/customers/{customer}/transactions', [\App\Http\Controllers\Api\V1\PosCustomerController::class, 'recordTransaction'])
                ->whereNumber('business')
                ->whereNumber('customer')
                ->name('pos.customers.transactions');

            Route::get('/{business}/customers/{customer}/ledger', [\App\Http\Controllers\Api\V1\PosCustomerController::class, 'ledger'])
                ->whereNumber('business')
                ->whereNumber('customer')
                ->name('pos.customers.ledger');

            // Order Management
            Route::get('/{business}/orders', [\App\Http\Controllers\Api\V1\PosController::class, 'indexOrders'])
                ->whereNumber('business')
                ->name('pos.orders.index');

            // Order Details
            Route::get('/{business}/orders/{order}', [\App\Http\Controllers\Api\V1\PosController::class, 'showOrder'])
                ->whereNumber('business')
                ->whereNumber('order')
                ->name('pos.orders.show');

            Route::post('/{business}/orders/{order}/summarize', [\App\Http\Controllers\Api\V1\PosController::class, 'summarizeOrder'])
                ->whereNumber('business')
                ->whereNumber('order')
                ->name('pos.orders.summarize');

            Route::post('/{business}/orders/{order}/fulfill', [\App\Http\Controllers\Api\V1\PosController::class, 'fulfillOrder'])
                ->whereNumber('business')
                ->whereNumber('order')
                ->name('pos.orders.fulfill');

            // Shift Management (Cash Drawer)
            Route::get('/{business}/shifts/current', [\App\Http\Controllers\Api\V1\PosShiftController::class, 'current'])
                ->whereNumber('business')
                ->name('pos.shifts.current');
            Route::get('/{business}/shifts', [\App\Http\Controllers\Api\V1\PosShiftController::class, 'index'])
                ->whereNumber('business')
                ->name('pos.shifts.index');
            Route::get('/{business}/shifts/{shift}', [\App\Http\Controllers\Api\V1\PosShiftController::class, 'show'])
                ->whereNumber('business')
                ->whereNumber('shift')
                ->name('pos.shifts.show');
            Route::post('/{business}/shifts/open', [\App\Http\Controllers\Api\V1\PosShiftController::class, 'open'])
                ->whereNumber('business')
                ->name('pos.shifts.open');
            Route::post('/{business}/shifts/{shift}/close', [\App\Http\Controllers\Api\V1\PosShiftController::class, 'close'])
                ->whereNumber('business')
                ->name('pos.shifts.close');

            // Refund/Returns
            Route::post('/{business}/refunds', [\App\Http\Controllers\Api\V1\PosRefundController::class, 'store'])
                ->whereNumber('business')
                ->name('pos.refunds.store');

            // Analytics
            Route::get('/{business}/inventory/health', [\App\Http\Controllers\Api\V1\PosController::class, 'inventoryHealth'])
                ->whereNumber('business')
                ->name('pos.inventory.health');

            Route::get('/{business}/analytics/dashboard', [\App\Http\Controllers\Api\V1\PosAnalyticsController::class, 'dashboard'])
                ->whereNumber('business')
                ->name('pos.analytics.dashboard');

            Route::get('/{business}/analytics/reports', [\App\Http\Controllers\Api\V1\PosAnalyticsController::class, 'reports'])
                ->whereNumber('business')
                ->name('pos.analytics.reports');

            // Expense Management
            Route::get('/{business}/expenses', [\App\Http\Controllers\Api\V1\PosExpenseController::class, 'index'])
                ->whereNumber('business')
                ->name('pos.expenses.index');
            Route::post('/{business}/expenses', [\App\Http\Controllers\Api\V1\PosExpenseController::class, 'store'])
                ->whereNumber('business')
                ->name('pos.expenses.store');
            Route::delete('/{business}/expenses/{expense}', [\App\Http\Controllers\Api\V1\PosExpenseController::class, 'destroy'])
                ->whereNumber('business')
                ->whereNumber('expense')
                ->name('pos.expenses.destroy');
        });
    });

    /*
    |--------------------------------------------------------------------------
    | Relworx Webhooks
    |--------------------------------------------------------------------------
    | Publicly accessible endpoint for receiving Relworx payment notifications.
    */
    Route::post('/webhooks/relworx', [RelworxWebhookController::class, 'handle'])->name('webhooks.relworx');

});

/*
|--------------------------------------------------------------------------
| Health Check Route
|--------------------------------------------------------------------------
*/
// Simple health check endpoint.
Route::get('/health', function () {
    return response()->json([
        'status' => 'ok',
        'timestamp' => now()->toIso8601String(),
    ]);
})->name('health');
