<?php

namespace Tests\Feature;

use App\Jobs\SendWelcomeEmail;
use App\Mail\WelcomeEmail;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Mail;
use Tests\TestCase;

class WelcomeEmailTest extends TestCase
{
    use RefreshDatabase;

    public function test_welcome_email_uses_custom_template_with_brand_colors()
    {
        Mail::fake();

        $user = User::factory()->create([
            'email' => 'test@example.com',
            'full_name' => 'Test User',
        ]);

        // Send the welcome email
        Mail::to($user->email)->send(new WelcomeEmail($user));

        // Assert the email was sent
        Mail::assertSent(WelcomeEmail::class, function ($mail) use ($user) {
            // Render the email to check its content
            $rendered = $mail->render();
            
            // Check for green and orange gradient colors
            $hasGreenOrangeGradient = str_contains($rendered, '#10b981') && 
                                     str_contains($rendered, '#f97316');
            
            // Check for custom welcome message
            $hasWelcomeMessage = str_contains($rendered, 'Welcome to');
            
            // Check for verification button
            $hasVerifyButton = str_contains($rendered, 'Verify Email Address');
            
            // Check that it's NOT the default Laravel template
            $notDefaultTemplate = !str_contains($rendered, 'background-color: #18181b');
            
            return $hasGreenOrangeGradient && 
                   $hasWelcomeMessage && 
                   $hasVerifyButton && 
                   $notDefaultTemplate &&
                   $mail->hasTo($user->email);
        });
    }

    public function test_welcome_email_job_is_dispatched_on_registration()
    {
        $response = $this->postJson('/api/v1/auth/register', [
            'full_name' => 'New User',
            'email' => 'newuser@example.com',
            'password' => 'Password@123',
            'password_confirmation' => 'Password@123',
        ]);

        $response->assertStatus(201);
        
        // Check that the job was dispatched
        $this->assertDatabaseHas('jobs', [
            'queue' => 'default',
        ]);
    }
}
