
    public function members(): View
    {
        $businessId = session('selected_business_id');
        
        if (!$businessId) {
            abort(404, 'No business selected');
        }

        $business = Business::where('id', $businessId)
            ->where('owner_user_id', Auth::guard('web')->id())
            ->firstOrFail();

        $members = $business->members()
            ->select('users.*')
            ->get()
            ->map(function (User $user) use ($business) {
                $membership = $business->memberships()->where('user_id', $user->id)->first();
                $user->role = $membership?->role;
                $user->joined_at = $membership?->joined_at;
                return $user;
            });
            
        $invitations = BusinessInvitation::where('business_id', $business->id)
            ->whereNull('accepted_at')
            ->whereNull('revoked_at')
            ->where(function ($q) {
                $q->whereNull('expires_at')->orWhere('expires_at', '>', now());
            })
            ->get();

        return view('web.business.members', compact('business', 'members', 'invitations'));
    }

    public function invite(Request $request): RedirectResponse
    {
        $businessId = session('selected_business_id');
        
        if (!$businessId) {
            return back()->withErrors(['error' => 'No business selected']);
        }

        $business = Business::where('id', $businessId)
            ->where('owner_user_id', Auth::guard('web')->id())
            ->firstOrFail();

        $request->validate([
            'email' => 'required|email',
            'role' => 'required|in:manager,editor',
        ]);

        $email = strtolower(trim($request->email));

        if (strtolower(Auth::guard('web')->user()->email) === $email) {
            return back()->withErrors(['email' => 'You cannot invite yourself.']);
        }

        $existingUser = User::where('email', $email)->first();
        if (!$existingUser) {
            return back()->withErrors(['email' => 'User must have an account.']);
        }

        if ($business->memberships()->where('user_id', $existingUser->id)->exists()) {
             return back()->withErrors(['email' => 'User is already a member.']);
        }

        $existingInvite = BusinessInvitation::where('business_id', $business->id)
            ->where('email', $email)
            ->whereNull('accepted_at')
            ->whereNull('revoked_at')
            ->where(function ($q) {
                $q->whereNull('expires_at')->orWhere('expires_at', '>', now());
            })
            ->first();

        if ($existingInvite) {
             return back()->withErrors(['email' => 'Invitation already sent.']);
        }

        $invite = BusinessInvitation::create([
            'business_id' => $business->id,
            'invited_by_user_id' => Auth::guard('web')->id(),
            'email' => $email,
            'role' => $request->role,
            'expires_at' => now()->addDays(7),
        ]);

        // Send Email (ignoring failure to not break flow, similar to API)
        try {
            Mail::to($existingUser->email)->send(new BusinessInvitationEmail(
                $business,
                $invite,
                Auth::guard('web')->user(),
                $existingUser
            ));
        } catch (\Throwable $e) {
            \Log::warning('Failed to send invite email', ['error' => $e->getMessage()]);
        }

        return back()->with('success', 'Invitation sent successfully.');
    }

    public function updateMemberRole(Request $request, $userId): RedirectResponse
    {
        $businessId = session('selected_business_id');
        
        if (!$businessId) {
            return back()->withErrors(['error' => 'No business selected']);
        }

        $business = Business::where('id', $businessId)
            ->where('owner_user_id', Auth::guard('web')->id())
            ->firstOrFail();

        if ((int)$business->owner_user_id === (int)$userId) {
             return back()->withErrors(['error' => 'Cannot change owner role.']);
        }
        
        $request->validate(['role' => 'required|in:manager,editor']);

        BusinessMembership::where('business_id', $business->id)
            ->where('user_id', $userId)
            ->update(['role' => $request->role]);

        return back()->with('success', 'Role updated successfully.');
    }

    public function removeMember($userId): RedirectResponse
    {
        $businessId = session('selected_business_id');
        
        if (!$businessId) {
            return back()->withErrors(['error' => 'No business selected']);
        }

        $business = Business::where('id', $businessId)
            ->where('owner_user_id', Auth::guard('web')->id())
            ->firstOrFail();

        if ((int)$business->owner_user_id === (int)$userId) {
             return back()->withErrors(['error' => 'Cannot remove owner.']);
        }

        BusinessMembership::where('business_id', $business->id)
            ->where('user_id', $userId)
            ->delete();

        return back()->with('success', 'Member removed successfully.');
    }

    public function transferOwnership(Request $request): RedirectResponse
    {
        $businessId = session('selected_business_id');
        
        if (!$businessId) {
            return back()->withErrors(['error' => 'No business selected']);
        }

        $business = Business::where('id', $businessId)
            ->where('owner_user_id', Auth::guard('web')->id())
            ->firstOrFail();

        $request->validate(['new_owner_email' => 'required|email']);
        
        $email = strtolower(trim($request->new_owner_email));
        $newOwner = User::where('email', $email)->first();

        if (!$newOwner) {
            return back()->withErrors(['new_owner_email' => 'User not found.']);
        }

        if ($newOwner->id === $business->owner_user_id) {
             return back()->withErrors(['new_owner_email' => 'User is already owner.']);
        }
        
        $oldOwnerId = $business->owner_user_id;

        // Create membership for new owner if not exists
        $membership = BusinessMembership::firstOrCreate([
            'business_id' => $business->id,
            'user_id' => $newOwner->id
        ], [
             'role' => BusinessMembership::ROLE_OWNER,
             'added_by_user_id' => Auth::guard('web')->id(),
             'joined_at' => now()
        ]);
        
        $membership->update(['role' => BusinessMembership::ROLE_OWNER]);

        // Downgrade old owner
        BusinessMembership::where('business_id', $business->id)
             ->where('user_id', $oldOwnerId)
             ->update(['role' => BusinessMembership::ROLE_MANAGER]);

        $business->update(['owner_user_id' => $newOwner->id]);

        return redirect()->route('web.profile')->with('success', 'Ownership transferred.');
    }
